package org.finconsgroup.itserr.criterion.common.service.impl;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.finconsgroup.itserr.criterion.common.service.CommonEmailService;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.mail.SimpleMailMessage;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.stereotype.Service;

import static org.finconsgroup.itserr.criterion.common.util.Constants.LOCAL;

@Slf4j
@Service
@RequiredArgsConstructor
public class DefaultEmailService implements CommonEmailService {

    // Automatically configured by Spring Boot from spring.mail.* properties 💖
    private final JavaMailSender mailSender;

    // Sender email address from YAML configuration
    @Value("${spring.email.from}")
    private String fromAddress;

    @Value("${spring.profiles.active:default}")
    private String activeProfile;

    @Override
    public boolean sendEmail(String receiver, String subject, String body) {
        try {
            // On local profile don't send mail
            if (activeProfile.equals(LOCAL.getValue())) {
                printEmailExample(receiver, subject, body);
                return true;
            }

            // Create simple email message
            SimpleMailMessage message = new SimpleMailMessage();
            message.setFrom(fromAddress);
            message.setTo(receiver);
            message.setSubject(subject);
            message.setText(body);

            // Send email through SMTP relay
            mailSender.send(message);
            log.info("Email sent successfully to: {}", receiver);
            return true;

        } catch (Exception e) {
            log.error("Failed to send email to: {}", receiver, e);
            return false;
        }
    }

    private void printEmailExample(String receiver, String subject, String body) {
        log.info("\n" +
                        "══════════════════════════════════════════════════════════════════════════\n" +
                        "                    EMAIL NOT SENT (LOCAL PROFILE)                        \n" +
                        "══════════════════════════════════════════════════════════════════════════\n" +
                        "To: {}\n" +
                        "Subject: {}\n" +
                        "──────────────────────────────────────────────────────────────────────────\n" +
                        "{}\n" +
                        "══════════════════════════════════════════════════════════════════════════",
                receiver, subject, body);
    }

}