package org.finconsgroup.itserr.criterion.common.http.interceptor;

import jakarta.servlet.http.HttpServletRequest;
import lombok.extern.slf4j.Slf4j;
import org.finconsgroup.itserr.criterion.common.http.config.PropagatingRestTemplateProperties;
import org.springframework.http.HttpRequest;
import org.springframework.http.client.ClientHttpRequestExecution;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.http.client.ClientHttpResponse;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import java.io.IOException;

@Slf4j
public class HeaderPropagatingInterceptor implements ClientHttpRequestInterceptor {

    private final PropagatingRestTemplateProperties properties;

    public HeaderPropagatingInterceptor(PropagatingRestTemplateProperties properties) {
        this.properties = properties;
    }

    @Override
    public ClientHttpResponse intercept(HttpRequest request,
                                        byte[] body,
                                        ClientHttpRequestExecution execution) throws IOException {

        // Propagate headers from incoming request
        propagateHeaders(request);

        // Add internal auth header
        addInternalAuthHeader(request);

        return execution.execute(request, body);
    }

    private void propagateHeaders(HttpRequest request) {
        ServletRequestAttributes attributes =
                (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();

        if (attributes == null) {
            return;
        }

        HttpServletRequest incomingRequest = attributes.getRequest();

        for (String headerName : properties.getPropagateHeaders()) {
            String headerValue = incomingRequest.getHeader(headerName);
            if (headerValue != null && !headerValue.isEmpty()) {
                request.getHeaders().set(headerName, headerValue);
                log.debug("Propagated header '{}' to outgoing request", headerName);
            }
        }
    }

    private void addInternalAuthHeader(HttpRequest request) {
        var internalConfig = properties.getInternal();

        if (!internalConfig.isEnabled()) {
            return;
        }

        if (internalConfig.getSecret() == null || internalConfig.getSecret().isEmpty()) {
            log.warn("Internal auth enabled but no secret configured!");
            return;
        }

        request.getHeaders().set(internalConfig.getHeaderName(), internalConfig.getSecret());
        log.debug("Added internal auth header to outgoing request");
    }
}