package org.finconsgroup.itserr.criterion.security.aspect;

import jakarta.servlet.http.HttpServletRequest;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.finconsgroup.itserr.criterion.common.exception.UnauthorizedException;
import org.finconsgroup.itserr.criterion.security.annotation.ExternalAuthCheck;
import org.finconsgroup.itserr.criterion.security.config.ExternalAuthProperties;
import org.finconsgroup.itserr.criterion.security.dto.AuthorizationRequest;
import org.finconsgroup.itserr.criterion.security.service.ExternalAuthService;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

@Aspect
@Slf4j
public class ExternalAuthAspect {

    private final ExternalAuthService externalAuthService;
    private final ExternalAuthProperties properties;

    public ExternalAuthAspect(ExternalAuthService externalAuthService,
                              ExternalAuthProperties properties) {
        this.externalAuthService = externalAuthService;
        this.properties = properties;
    }

    @Around("@annotation(externalAuthCheck)")
    public Object checkAuthorizationOnMethod(ProceedingJoinPoint joinPoint,
                                             ExternalAuthCheck externalAuthCheck) throws Throwable {
        return doCheck(joinPoint, externalAuthCheck);
    }

    @Around("@within(externalAuthCheck) && !@annotation(org.finconsgroup.itserr.criterion.security.annotation.ExternalAuthCheck)")
    public Object checkAuthorizationOnClass(ProceedingJoinPoint joinPoint,
                                            ExternalAuthCheck externalAuthCheck) throws Throwable {
        return doCheck(joinPoint, externalAuthCheck);
    }

    private Object doCheck(ProceedingJoinPoint joinPoint,
                           ExternalAuthCheck externalAuthCheck) throws Throwable {

        // Skip if disabled globally
        if (!properties.isEnabled()) {
            log.debug("External auth check is disabled globally, proceeding without check");
            return joinPoint.proceed();
        }

        HttpServletRequest request = getCurrentRequest();

        // Check if this is a valid internal call
        if (isValidInternalCall(request)) {
            log.debug("Valid internal call detected, skipping external auth check");
            return joinPoint.proceed();
        }

        // External call - perform full auth check
        AuthorizationRequest authRequest = buildAuthRequest(request, joinPoint);

        log.debug("External call - checking authorization for {} {}",
                authRequest.getHttpMethod(),
                authRequest.getRequestUri());

        boolean isAuthorized = externalAuthService.checkAuthorization(
                externalAuthCheck.authEndpoint(),
                authRequest
        );

        if (!isAuthorized) {
            log.warn("Authorization DENIED for {} {} from {}",
                    authRequest.getHttpMethod(),
                    authRequest.getRequestUri(),
                    authRequest.getRemoteAddress());
            throw new UnauthorizedException(externalAuthCheck.message());
        }

        log.debug("Authorization GRANTED for {} {}",
                authRequest.getHttpMethod(),
                authRequest.getRequestUri());

        return joinPoint.proceed();
    }

    /**
     * Check if the request is a valid internal call from another microservice.
     */
    private boolean isValidInternalCall(HttpServletRequest request) {
        if (request == null) {
            return false;
        }

        var internalConfig = properties.getInternal();

        if (!internalConfig.isConfigured()) {
            return false;
        }

        String internalAuthHeader = request.getHeader(internalConfig.getHeaderName());

        if (internalAuthHeader == null || internalAuthHeader.isEmpty()) {
            return false;
        }

        boolean isValid = internalConfig.getSecret().equals(internalAuthHeader);

        if (!isValid) {
            log.warn("Invalid internal auth header received - possible security breach attempt!");
        }

        return isValid;
    }

    private HttpServletRequest getCurrentRequest() {
        ServletRequestAttributes attrs =
                (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
        return attrs != null ? attrs.getRequest() : null;
    }

    private AuthorizationRequest buildAuthRequest(HttpServletRequest request,
                                                  ProceedingJoinPoint joinPoint) {
        MethodSignature signature = (MethodSignature) joinPoint.getSignature();

        AuthorizationRequest.AuthorizationRequestBuilder builder = AuthorizationRequest.builder()
                .targetMethod(signature.getMethod().getName())
                .targetClass(signature.getDeclaringTypeName());

        if (request != null) {
            builder
                    .authorizationHeader(request.getHeader("Authorization"))
                    .requestUri(request.getRequestURI())
                    .httpMethod(request.getMethod())
                    .remoteAddress(getClientIpAddress(request))
                    .userAgent(request.getHeader("User-Agent"))
                    .correlationId(request.getHeader("X-Correlation-ID"));
        }

        return builder.build();
    }

    private String getClientIpAddress(HttpServletRequest request) {
        String xForwardedFor = request.getHeader("X-Forwarded-For");
        if (xForwardedFor != null && !xForwardedFor.isEmpty()) {
            return xForwardedFor.split(",")[0].trim();
        }
        return request.getRemoteAddr();
    }
}