package org.finconsgroup.itserr.criterion.security.health;

import io.github.resilience4j.circuitbreaker.CircuitBreaker;
import org.finconsgroup.itserr.criterion.security.service.ExternalAuthService;
import org.springframework.boot.actuate.health.Health;
import org.springframework.boot.actuate.health.HealthIndicator;

/**
 * Health Indicator for monitoring external authorization status.
 * Exposes Circuit Breaker information via /actuator/health endpoint.
 *
 * <p>Health status mapping:</p>
 * <ul>
 *   <li>CLOSED → UP (service is healthy)</li>
 *   <li>HALF_OPEN → UNKNOWN (service is being tested)</li>
 *   <li>OPEN → DOWN (service is unavailable)</li>
 * </ul>
 */
public class ExternalAuthHealthIndicator implements HealthIndicator {

    private final ExternalAuthService externalAuthService;

    public ExternalAuthHealthIndicator(ExternalAuthService externalAuthService) {
        this.externalAuthService = externalAuthService;
    }

    @Override
    public Health health() {
        CircuitBreaker.State state = externalAuthService.getCircuitBreakerState();
        CircuitBreaker.Metrics metrics = externalAuthService.getCircuitBreakerMetrics();

        Health.Builder builder = switch (state) {
            case CLOSED -> Health.up();
            case HALF_OPEN -> Health.unknown();
            case OPEN, DISABLED, FORCED_OPEN, METRICS_ONLY -> Health.down();
        };

        return builder
                .withDetail("circuitBreakerState", state.name())
                .withDetail("failureRate", String.format("%.2f%%", metrics.getFailureRate()))
                .withDetail("successfulCalls", metrics.getNumberOfSuccessfulCalls())
                .withDetail("failedCalls", metrics.getNumberOfFailedCalls())
                .withDetail("notPermittedCalls", metrics.getNumberOfNotPermittedCalls())
                .withDetail("bufferedCalls", metrics.getNumberOfBufferedCalls())
                .build();
    }
}