package org.finconsgroup.itserr.criterion.security.annotation;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Annotation to enable external authorization check on REST endpoints.
 * Can be applied to individual methods or entire controller classes.
 *
 * <p>The annotated endpoint will call an external authorization service
 * before proceeding. If the service returns true, the request continues;
 * otherwise, a 401 Unauthorized response is returned.</p>
 *
 * <p>Usage example:</p>
 * <pre>
 * &#64;RestController
 * &#64;RequestMapping("/api/orders")
 * public class OrderController {
 *
 *     &#64;GetMapping
 *     &#64;ExternalAuthCheck
 *     public List&lt;Order&gt; findAll() {
 *         return orderService.findAll();
 *     }
 * }
 * </pre>
 */
@Target({ElementType.METHOD, ElementType.TYPE})
@Retention(RetentionPolicy.RUNTIME)
public @interface ExternalAuthCheck {

    /**
     * Custom authorization endpoint URL.
     * If empty, uses the default URL from configuration (wp3.external-auth.url).
     */
    String authEndpoint() default "";

    /**
     * Custom error message returned when authorization is denied.
     */
    String message() default "Unauthorized - External authorization failed";
}