package com.finconsgroup.itserr.marketplace.audit.dm.controller;

import com.finconsgroup.itserr.marketplace.audit.dm.api.AuditLogApi;
import com.finconsgroup.itserr.marketplace.audit.dm.dto.InputCreateAuditLogDto;
import com.finconsgroup.itserr.marketplace.audit.dm.dto.OutputAuditLogDto;
import com.finconsgroup.itserr.marketplace.audit.dm.service.AuditLogService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.propertyeditors.StringTrimmerEditor;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.RestController;

import java.util.UUID;

/**
 * REST controller class for handling user audit related API requests.
 *
 * <p>Implements the {@link AuditLogApi} interface.</p>
 */
@Slf4j
@RequiredArgsConstructor
@RestController
public class AuditLogController implements AuditLogApi {

    private final AuditLogService auditLogService;

    @InitBinder
    public void initBinder(WebDataBinder binder) {
        // trim strings before triggering validation, but do not convert to null
        // as otherwise it throws - MissingServletRequestParameterException
        binder.registerCustomEditor(String.class, new StringTrimmerEditor(false));
    }

    @Override
    public Page<OutputAuditLogDto> findAll(
            final int pageNumber,
            final int pageSize,
            final String sort,
            final Sort.Direction direction
    ) {
        Pageable pageable = PageRequest.of(pageNumber, pageSize, direction, sort);
        return auditLogService.findAll(pageable);
    }

    @Override
    public OutputAuditLogDto findById(final UUID auditId) {
        return auditLogService.findById(auditId);
    }

    @Override
    public OutputAuditLogDto createAuditLog(final InputCreateAuditLogDto inputCreateAuditLogDto) {
        return auditLogService.create(inputCreateAuditLogDto);
    }
}
