package com.finconsgroup.itserr.marketplace.discussion.bs.controller;

import com.finconsgroup.itserr.marketplace.discussion.bs.api.DiscussionApi;
import com.finconsgroup.itserr.marketplace.discussion.bs.dto.DiscussionDTO;
import com.finconsgroup.itserr.marketplace.discussion.bs.dto.InputUpdateDiscussionDto;
import com.finconsgroup.itserr.marketplace.discussion.bs.dto.ThreadDTO;
import com.finconsgroup.itserr.marketplace.discussion.bs.enums.ReactionType;
import com.finconsgroup.itserr.marketplace.discussion.bs.service.DiscussionService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Sort;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.RestController;

/**
 * REST controller that exposes endpoints for managing discussions, their threads,
 * and reactions.
 *
 * <p>
 * This controller implements the HTTP contract defined by {@link DiscussionApi}
 * and delegates all business operations to the {@link DiscussionService}.
 * As such, the controller itself contains no business logic; it simply
 * orchestrates request handling and forwards calls to the service layer.
 * </p>
 */
@RestController
@Slf4j
@RequiredArgsConstructor
public class DiscussionController implements DiscussionApi {

    private final DiscussionService discussionService;

    /**
     * Retrieves a paginated list of discussions using the service layer.
     *
     * @param page           page number (0-based)
     * @param size           page size
     * @param sortBy         field to sort by (e.g. createdAt)
     * @param direction      sort direction (asc|desc)
     * @param visibility     optional visibility filter (e.g. public/private)
     * @param includeThreads whether to include discussion threads in the response
     * @return a page of {@link DiscussionDTO} results (may be enriched downstream)
     */
    @Override
    public Page<DiscussionDTO> getDiscussions(int page, int size, String sortBy, String direction, String visibility, boolean includeThreads) {
        return discussionService.getDiscussions(page, size, sortBy, direction, visibility, includeThreads);
    }

    /**
     * Creates a new discussion by delegating to the service layer.
     *
     * @param discussionDTO the payload describing the discussion to create
     * @return a response indicating creation result (typically 201 Created)
     */
    @Override
    public ResponseEntity<Void> createDiscussion(DiscussionDTO discussionDTO) {
        return discussionService.createDiscussion(discussionDTO);
    }

    /**
     * Updates an existing discussion with new details.
     *
     * @param id the identifier of the discussion to be updated
     * @param discussionDto the payload containing updated discussion details
     * @return a {@link ResponseEntity} containing the updated {@link DiscussionDTO}
     */
    @Override
    public ResponseEntity<DiscussionDTO> updateDiscussion(String id, InputUpdateDiscussionDto discussionDto) {
        log.info("PATCH discussion with id: {}", id);
        DiscussionDTO discussionDTO = discussionService.updateDiscussion(id, discussionDto);
        return new ResponseEntity<>(discussionDTO, HttpStatus.OK);
    }

    /**
     * Deletes a discussion by its identifier.
     *
     * @param id the discussion identifier
     * @return a response indicating deletion result (typically 204 No Content)
     */
    @Override
    public ResponseEntity<Void> deleteDiscussion(String id) {
        return discussionService.deleteDiscussion(id);
    }

    @Override
    public Page<ThreadDTO> getThreadsByDiscussionId(String discussionId, int page, int size, String sortBy, Sort.Direction direction) {
        return discussionService.getThreadsByDiscussionId(discussionId, page, size, sortBy, direction);
    }

    /**
     * Adds a new thread to an existing discussion.
     *
     * @param discussionId the discussion identifier
     * @param threadDTO    the thread payload to add
     * @return a response indicating creation result (typically 201 Created)
     */
    @Override
    public ResponseEntity<ThreadDTO> addThread(String discussionId, ThreadDTO threadDTO) {
        return discussionService.addThread(discussionId, threadDTO);
    }

    /**
     * Deletes a thread from a discussion.
     *
     * @param discussionId the discussion identifier
     * @param threadId     the thread identifier
     * @return a response indicating deletion result (typically 204 No Content)
     */
    @Override
    public ResponseEntity<Void> deleteThread(String discussionId, String threadId) {
        return discussionService.deleteThread(discussionId, threadId);
    }

    /**
     * Updates an existing thread within a discussion.
     *
     * @param discussionId the discussion identifier
     * @param threadId     the thread identifier
     * @param threadDTO    the updated thread payload
     * @return a response indicating update result (e.g. 200 OK or 204 No Content)
     */
    @Override
    public ResponseEntity<ThreadDTO> updateThread(String discussionId, String threadId, ThreadDTO threadDTO) {
        return discussionService.updateThread(discussionId, threadId, threadDTO);
    }

    /**
     * Adds a reaction to a discussion on behalf of a user.
     *
     * @param discussionId the discussion identifier
     * @param userId       the user applying the reaction
     * @param reactionType the {@link ReactionType} to apply
     * @return a response indicating operation result (e.g. 200/201)
     */
    @Override
    public ResponseEntity<Void> addReactionToDiscussion(String discussionId, String userId, ReactionType reactionType) {
        return discussionService.addReactionToDiscussion(discussionId, userId, reactionType);
    }

    /**
     * Removes a user's reaction from a discussion.
     *
     * @param discussionId the discussion identifier
     * @param userId       the user whose reaction should be removed
     * @return a response indicating deletion result (typically 204 No Content)
     */
    @Override
    public ResponseEntity<Void> removeReactionFromDiscussion(String discussionId, String userId) {
        return discussionService.removeReactionFromDiscussion(discussionId, userId);
    }

    /**
     * Adds a reaction to a specific thread within a discussion.
     *
     * @param discussionId the discussion identifier
     * @param threadId     the thread identifier
     * @param userId       the user applying the reaction
     * @param reactionType the {@link ReactionType} to apply
     * @return a response indicating operation result (e.g. 200/201)
     */
    @Override
    public ResponseEntity<Void> addReactionToThread(String discussionId, String threadId, String userId, ReactionType reactionType) {
        return discussionService.addReactionToThread(discussionId, threadId, userId, reactionType);
    }

    /**
     * Removes a user's reaction from a thread.
     *
     * @param discussionId the discussion identifier
     * @param threadId     the thread identifier
     * @param userId       the user whose reaction should be removed
     * @return a response indicating deletion result (typically 204 No Content)
     */
    @Override
    public ResponseEntity<Void> removeReactionFromThread(String discussionId, String threadId, String userId) {
        return discussionService.removeReactionFromThread(discussionId, threadId, userId);
    }

    @Override
    public DiscussionDTO getDiscussionByResourceId(String resourceId) {
        return discussionService.getDiscussionByResourceId(resourceId);
    }

}
