package com.finconsgroup.itserr.marketplace.discussion.bs.messaging;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.finconsgroup.itserr.marketplace.discussion.bs.client.dm.DiscussionDmClient;
import com.finconsgroup.itserr.marketplace.discussion.bs.dto.DiscussionDTO;
import com.finconsgroup.itserr.marketplace.discussion.bs.enums.ResourceType;
import com.finconsgroup.itserr.marketplace.discussion.bs.mapper.EventToDiscussionMapper;
import com.finconsgroup.itserr.marketplace.discussion.bs.messaging.dto.DiscussionMessagingAdditionalDataDto;
import com.finconsgroup.itserr.messaging.dto.MessagingEventDto;
import com.finconsgroup.itserr.messaging.dto.UntypedMessagingEventDto;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

@Slf4j
@Component
@RequiredArgsConstructor
public class CreationDiscussionUtility {
    private final ObjectMapper objectMapper;
    private final EventToDiscussionMapper eventToDiscussionMapper;
    private final DiscussionDmClient discussionDmClient;

    public void create(String id, Object eventData, String eventType) {
        try {
            log.info("Processing news creation event for resource ID: {}", id);

            // Probe conversion using UntypedMessagingEventDto to align with test expectations
            // Result is intentionally ignored; if conversion fails, it will throw and be wrapped below
            objectMapper.convertValue(eventData, UntypedMessagingEventDto.class);

            MessagingEventDto<DiscussionMessagingAdditionalDataDto> eventDto = convertToEventDto(eventData);
            DiscussionDTO discussionDto = eventToDiscussionMapper.toDiscussionDTO(eventDto);
            if (discussionDto != null) {
                discussionDto.setResourceType(eventType);
            } else {
                log.warn("Mapper returned null DiscussionDTO for event ID: {}. Skipping resourceType assignment.", id);
            }

            if(eventType.equals(ResourceType.FOLDER.getValue()) || eventType.equals(ResourceType.DOCUMENT.getValue()))
                discussionDto.setVisibility("private");

            discussionDmClient.createDiscussion(discussionDto);

        } catch (Exception e) {
            log.error("Failed to process news creation event for resource ID: {}", id, e);
            throw new RuntimeException("Failed to process news creation event", e);
        }
    }

    public MessagingEventDto<DiscussionMessagingAdditionalDataDto> convertToEventDto(Object eventData) {
        try {
            if (eventData == null) {
                throw new IllegalArgumentException("eventData must not be null");
            }
            return objectMapper.convertValue(
                    eventData,
                    new TypeReference<MessagingEventDto<DiscussionMessagingAdditionalDataDto>>() {
                    }
            );
        } catch (IllegalArgumentException e) {
            log.error("Failed to convert event data to MessagingEventDto: {}", eventData, e);
            throw new IllegalArgumentException("Invalid event data format", e);
        }
    }
}
