package com.finconsgroup.itserr.marketplace.discussion.dm.controller;

import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.discussion.dm.api.DiscussionApi;
import com.finconsgroup.itserr.marketplace.discussion.dm.dto.DiscussionDTO;
import com.finconsgroup.itserr.marketplace.discussion.dm.dto.InputUpdateDiscussionDto;
import com.finconsgroup.itserr.marketplace.discussion.dm.dto.ThreadDTO;
import com.finconsgroup.itserr.marketplace.discussion.dm.enums.ReactionType;
import com.finconsgroup.itserr.marketplace.discussion.dm.service.DiscussionService;
import lombok.RequiredArgsConstructor;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import java.util.List;
import java.util.UUID;

/**
 * REST Controller for managing discussions, threads, and reactions.
 * This controller exposes HTTP endpoints for all discussion-related operations.
 */
@RestController
@RequiredArgsConstructor
public class DiscussionController implements DiscussionApi {

    private final DiscussionService discussionService;

    /**
     * Retrieves a discussion by its unique identifier.
     *
     * @param id the unique identifier of the discussion
     * @return the discussion DTO
     */
    public DiscussionDTO getDiscussionById(String id) {
        return discussionService.getDiscussionById(id);
    }

    /**
     * Retrieves a discussion associated with the specified resource identifier.
     *
     * @param resourceId the unique identifier of the resource
     * @return the discussion DTO associated with the resource
     */
    public DiscussionDTO getDiscussionByResourceId(String resourceId) {
        return discussionService.getDiscussionByResourceId(resourceId);
    }

    /**
     * Retrieves a paginated list of discussions.
     *
     * @param page the page number (default: 0)
     * @param size the page size (default: 10)
     * @param sortBy the field to sort by (default: createdAt)
     * @param direction the sort direction (default: desc)
     * @param visibility the visibility filter (optional)
     * @param includeThreads whether to include threads in the response (default: true)
     * @return a paginated result of discussions
     */
    public Page<DiscussionDTO> getDiscussions(int page, int size, String sortBy, Sort.Direction direction, String visibility, boolean includeThreads) {
        Pageable pageable = PageRequest.of(page, size, direction, sortBy);

        return discussionService.getDiscussions(pageable, visibility, includeThreads);
    }

    /**
     * Creates a new discussion.
     *
     * @param discussionDTO the discussion data to create
     * @return response entity indicating the result of the operation
     */
    public ResponseEntity<Void> createDiscussion(DiscussionDTO discussionDTO) {
        return discussionService.createDiscussion(discussionDTO);
    }

    /**
     * Deletes a discussion by its unique identifier.
     *\
     * @param id the unique identifier of the discussion to delete
     * @return response entity indicating the result of the operation
     */
    public ResponseEntity<Void> deleteDiscussion(String id) {
        return discussionService.deleteDiscussion(id);
    }

    /**
     * Updates a discussion by its unique identifier.
     *
     * @param discussionId the unique identifier of the discussion to update
     * @param inputUpdateDiscussionDto the discussion data to update
     * @return the updated discussion DTO
     */
    public DiscussionDTO updateDiscussion(UUID discussionId, InputUpdateDiscussionDto inputUpdateDiscussionDto) {
        String currentUser = JwtTokenHolder.getPreferredUsernameOrThrow();
        return discussionService.updateDiscussion(discussionId, inputUpdateDiscussionDto, currentUser);
    }

    /**
     * Retrieves a paginated list of threads associated with a discussion.
     *
     * @param discussionId the unique identifier of the discussion
     * @param page the page number (default: 0)
     * @param size the page size (default: 10)
     * @param sortBy the field to sort by (default: createdAt)
     * @param direction the sort direction (default: desc)
     * @return a paginated result of thread DTOs
     */
    public Page<ThreadDTO> getThreadsByDiscussionId(String discussionId, int page, int size, String sortBy, Sort.Direction direction) {
        Pageable pageable = PageRequest.of(page, size, direction, sortBy);
        return discussionService.getThreadsByDiscussionId(discussionId, pageable);
    }

    /**
     * Adds a new thread to an existing discussion.
     *
     * @param discussionId the unique identifier of the discussion
     * @param threadDTO the thread data to add
     * @return response entity indicating the result of the operation
     */
    public ResponseEntity<ThreadDTO> addThread(String discussionId, ThreadDTO threadDTO) {
        return discussionService.addThread(discussionId, threadDTO);
    }

    /**
     * Deletes a specific thread from a discussion.
     *
     * @param discussionId the unique identifier of the discussion
     * @param threadId the unique identifier of the thread to delete
     * @return response entity indicating the result of the operation
     */
    public ResponseEntity<Void> deleteThread(String discussionId, String threadId) {
        return discussionService.deleteThread(discussionId, threadId);
    }

    /**
     * Updates a specific thread within a discussion.
     *
     * @param discussionId the unique identifier of the discussion
     * @param threadId the unique identifier of the thread to update
     * @param threadDTO the thread data to update
     * @return response entity indicating the result of the operation
     */
    public ResponseEntity<ThreadDTO> updateThread(String discussionId, String threadId, ThreadDTO threadDTO) {
        return discussionService.updateThread(discussionId, threadId, threadDTO);
    }

    /**
     * Adds a reaction to a discussion by a user.
     *
     * @param discussionId the unique identifier of the discussion
     * @param userId the unique identifier of the user
     * @param reactionType the type of reaction to add
     * @return response entity indicating the result of the operation
     */
    public ResponseEntity<Void> addReactionToDiscussion(String discussionId, String userId, ReactionType reactionType) {
        return discussionService.addReactionToDiscussion(discussionId, userId, reactionType);
    }

    /**
     * Removes a reaction from a discussion by a user.
     *
     * @param discussionId the unique identifier of the discussion
     * @param userId the unique identifier of the user
     * @return response entity indicating the result of the operation
     */
    public ResponseEntity<Void> removeReactionFromDiscussion(String discussionId, String userId) {
        return discussionService.removeReactionFromDiscussion(discussionId, userId);
    }

    /**
     * Adds a reaction to a specific thread within a discussion.
     *
     * @param discussionId the unique identifier of the discussion
     * @param threadId the unique identifier of the thread
     * @param userId the unique identifier of the user
     * @param reactionType the type of reaction to add
     * @return response entity indicating the result of the operation
     */
    public ResponseEntity<Void> addReactionToThread(String discussionId, String threadId, String userId, ReactionType reactionType) {
        return discussionService.addReactionToThread(discussionId, threadId, userId, reactionType);
    }

    /**
     * Removes a reaction from a specific thread within a discussion.
     *
     * @param discussionId the unique identifier of the discussion
     * @param threadId the unique identifier of the thread
     * @param userId the unique identifier of the user
     * @return response entity indicating the result of the operation
     */
    public ResponseEntity<Void> removeReactionFromThread(String discussionId, String threadId, String userId) {
        return discussionService.removeReactionFromThread(discussionId, threadId, userId);
    }
}
