package com.finconsgroup.itserr.marketplace.discussion.dm.entity;

import jakarta.persistence.*;
import lombok.*;
import java.time.Instant;
import java.util.List;
import java.util.UUID;

@Entity
@Table(name = "discussions")
@NoArgsConstructor
@AllArgsConstructor
@Builder
@Data
public class Discussion {
    /**
     * Represents the unique identifier for the discussion entity.
     * This value is automatically generated using the UUID generation strategy.
     * It serves as the primary identifier for the entity in the database.
     */
    @Id
    @GeneratedValue(strategy = GenerationType.UUID)
    private UUID id;
    /**
     * The title of the discussion.
     */
    private String title;
    /**
     * Represents the textual content of the discussion, stored as a large text field in the database.
     */
    @Column(columnDefinition = "TEXT")
    private String content;
    /**
     * Represents the identifier of the user who created the discussion.
     */
    private String createdBy;
    /**
     * The date and time when the discussion entity was created.
     * This field stores the timestamp for the creation
     * of a discussion record in the system.
     */
    private Instant createdAt;
    /**
     * Represents the identifier of an external resource associated with the discussion.
     * This could be used to reference a related entity or resource in the system.
     */
    private UUID resourceId;
    /**
     * Represents the type of the external resource associated with the discussion.
     * This field categorizes the kind of resource that the discussion is related to.
     */
    private String resourceType;
    /**
     * Represents the visibility setting of a discussion.
     * This variable determines the access level or scope
     * of visibility for the discussion entity.
     */
    private String visibility;
    /**
     * Represents the URL associated with the discussion.
     * This field can store a web link or reference URL
     * related to the discussion content.
     */
    private String url;

    /**
     * Represents the identifier of the user who last updated the discussion.
     */
    private String updatedBy;

    /**
     * The date and time when the discussion entity was last updated.
     * This field stores the timestamp for the last modification
     * of a discussion record in the system.
     */
    private Instant updatedAt;

    /**
     * Represents the list of threads associated with a discussion.
     * Each thread belongs to a specific discussion, and the relationship
     * is mapped through the "discussion" field in the Thread entity.
     *
     * This relationship is a one-to-many association, where a discussion can
     * have multiple threads. The cascade operations and orphan removal
     * allow automatic persistence, deletion, and removal of threads in
     * relation to the discussion.
     *
     * - CascadeType.ALL ensures all related thread operations are
     *   cascaded to the associated discussion.
     * - orphanRemoval=true automatically removes threads from the database
     *   if they are removed from the collection.
     */
    @OneToMany(mappedBy = "discussion", cascade = CascadeType.ALL, orphanRemoval = true)
    private List<Thread> threads;

    /**
     * Represents a collection of reactions associated with the {@code Discussion} entity.
     * Each reaction in the list corresponds to a distinct user interaction with the discussion,
     * such as a like or an emoji-based reaction.
     *
     * The relationship is defined as a one-to-many association with the {@code DiscussionReaction} entity.
     * The {@code reactions} are fully managed by the {@code Discussion} entity, such that:
     *
     * - If a {@code Discussion} entity is deleted, all associated {@code DiscussionReaction} entities
     *   will also be removed from the database due to the {@code CascadeType.ALL} configuration.
     * - If an individual {@code DiscussionReaction} is removed from the list, it will also be
     *   deleted from the database because of the {@code orphanRemoval = true} setting.
     *
     * Mapped by the {@code discussion} field in the {@code DiscussionReaction} entity.
     */
    @OneToMany(mappedBy = "discussion", cascade = CascadeType.ALL, orphanRemoval = true)
    private List<DiscussionReaction> reactions;

    /**
     * Callback method annotated with {@code @PrePersist} that is executed
     * automatically before the entity is persisted to the database.
     * This method initializes the {@code createdAt} field with the current
     * date and time, ensuring that the creation timestamp is captured
     * when a new entity is inserted into the database.
     */
    @PrePersist
    public void onCreate() {
        createdAt = Instant.now();
        updatedAt = Instant.now();
    }

    /**
     * Callback method annotated with {@code @PreUpdate} that is executed
     * automatically before the entity is updated in the database.
     * This method updates the {@code updatedAt} field with the current
     * date and time, ensuring that the last modification timestamp is captured
     * when an entity is updated in the database.
     */
    @PreUpdate
    public void onUpdate() {
        updatedAt = Instant.now();
    }
}
