package com.finconsgroup.itserr.marketplace.discussion.dm.entity;

import jakarta.persistence.*;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;
import java.time.Instant;
import java.util.List;
import java.util.UUID;

@Entity
@Table(name = "threads")
@NoArgsConstructor
@AllArgsConstructor
@Builder
@Data
public class Thread {
    /**
     * Represents the universally unique identifier (UUID) for the thread entity.
     * This value is automatically generated using the UUID generation strategy
     * and serves as the primary key for identifying the thread in the database.
     */
    @Id
    @GeneratedValue(strategy = GenerationType.UUID)
    private UUID id;

    /**
     * Represents the association between the {@code Thread} entity and the {@code Discussion} entity.
     * This relationship is defined as a many-to-one association, where multiple {@code Thread} entities
     * can belong to a single {@code Discussion}.
     *
     * Key details about the relationship:
     * - The foreign key column name is defined as {@code discussion_id}.
     * - The {@code nullable = false} constraint ensures that a thread must be associated with a discussion.
     * - {@code fetch = FetchType.LAZY} is used to indicate that the associated {@code Discussion} entity
     *   will be fetched lazily, meaning it will only be retrieved when explicitly accessed.
     */
    @ManyToOne(fetch = FetchType.LAZY)
    @JoinColumn(name = "discussion_id", nullable = false)
    private Discussion discussion;

    /**
     * Represents the parent thread in a thread hierarchy.
     * This field establishes a self-referential many-to-one relationship,
     * where a thread may have another thread as its parent.
     *
     * It is mapped to the "parent_id" column in the database and is fetched lazily.
     * This relationship allows the creation of nested or hierarchical thread structures.
     */
    @ManyToOne(fetch = FetchType.LAZY)
    @JoinColumn(name = "parent_id")
    private Thread parent;

    /**
     * Represents a list of replies associated with the current thread.
     * This defines a self-referencing one-to-many relationship where a parent thread
     * can have multiple child threads as replies. Each reply is mapped to its parent
     * thread through the "parent" field in the {@code Thread} entity.
     *
     * CascadeType.ALL ensures that all cascaded operations (persist, merge, remove,
     * refresh, detach) on the parent thread will also be applied to its replies.
     *
     * The relationship is bidirectional, where the "parent" field in the child
     * thread refers back to the parent thread.
     */
    @OneToMany(mappedBy = "parent", cascade = CascadeType.ALL)
    private List<Thread> replies;

    /**
     * Represents the identifier of the user who created the thread.
     * This value corresponds to the user entity's unique identifier
     * and is utilized to track the creator of the thread in the system.
     */
    private String createdBy;
    /**
     * Represents the timestamp of when the Thread entity was created.
     * This field is automatically populated with the current date and time
     * when the entity is persisted to the database.
     *
     * The value is set using the {@code @PrePersist} callback method {@code onCreate},
     * ensuring that a creation timestamp is assigned to every newly created Thread record.
     */
    private Instant createdAt;

    /**
     * Timestamp of the last modification to the thread's content. Null if never modified.
     */
    @Column(name = "modified_at")
    private Instant modifiedAt;

    /**
     * Represents the textual content of a thread.
     * This is intended to store detailed textual data, with its database column
     * defined as a large text field (using the "TEXT" type) to accommodate varying lengths of content.
     *
     * It allows threads to contain textual information, such as messages, comments,
     * or descriptions, providing a key aspect of participation in discussions.
     */
    @Column(columnDefinition = "TEXT")
    private String content;

    /**
     * Represents the list of user reactions associated with a specific thread in the discussion system.
     * Each reaction in the list encapsulates a user's interaction with the thread, such as a like
     * or emoji-based reaction.
     *
     * This relationship is defined as a one-to-many association with the {@code ThreadReaction} entity:
     * - Each {@code ThreadReaction} is linked to a single thread via the {@code thread} field in
     *   the {@code ThreadReaction} entity.
     * - A thread can have multiple reactions, capturing different interactions by various users.
     *
     * Key configurations:
     * - CascadeType.ALL: Ensures that all persistence operations (e.g., persist, merge, remove)
     *   executed on the {@code Thread} entity are cascaded to its associated {@code ThreadReaction} entities.
     * - orphanRemoval = true: Automatically removes reactions from the database if they are
     *   removed from the collection in the {@code Thread} entity.
     *
     * This field is mapped by the {@code thread} property in the {@code ThreadReaction} entity.
     */
    @OneToMany(mappedBy = "thread", cascade = CascadeType.ALL, orphanRemoval = true)
    private List<ThreadReaction> reactions;

    /**
     * Callback method annotated with {@code @PrePersist}, which is automatically
     * invoked by the JPA provider before the entity is persisted to the database.
     *
     * This method sets the {@code createdAt} field to the current date and time,
     * ensuring that the creation timestamp is recorded when a new instance
     * of the entity is created and saved.
     */
    @PrePersist
    public void onCreate() {
        createdAt = Instant.now();
    }


}
