package com.finconsgroup.itserr.marketplace.discussion.dm.service;

import com.finconsgroup.itserr.marketplace.discussion.dm.bean.DiscussionFilterRequest;
import com.finconsgroup.itserr.marketplace.discussion.dm.dto.DiscussionDTO;
import com.finconsgroup.itserr.marketplace.discussion.dm.dto.InputUpdateDiscussionDto;
import com.finconsgroup.itserr.marketplace.discussion.dm.dto.ThreadDTO;
import com.finconsgroup.itserr.marketplace.discussion.dm.enums.ReactionType;
import org.springframework.data.domain.Page;
import org.springframework.http.ResponseEntity;

import org.springframework.data.domain.Pageable;

import java.util.UUID;

public interface DiscussionService {
    /**
     * Retrieves a discussion associated with the specified resource identifier.
     *
     * @param resourceId the unique identifier of the resource for which the discussion is to be retrieved
     * @param resourceType the type of the resource for which the discussion is to be retrieved
     * @return a {@link DiscussionDTO} representing the discussion associated with the given resource identifier,
     * or null if no discussion is found
     */
    DiscussionDTO getDiscussionForResource(String resourceId, String resourceType);
    /**
     * Retrieves a discussion by its unique identifier.
     *
     * @param id the unique identifier (UUID as a string) of the discussion to retrieve
     * @param includeThreads whether to include threads in the response (default: true)
     * @return the discussion associated with the provided identifier as a DiscussionDTO object
     */
    DiscussionDTO getDiscussionById(String id, boolean includeThreads);

    /**
     * Creates a new discussion based on the provided discussion data.
     *
     * @param discussionDTO the data transfer object containing details of the discussion to be created
     * @return the created {@link DiscussionDTO}
     */
    DiscussionDTO createDiscussion(DiscussionDTO discussionDTO);
    /**
     * Deletes a discussion identified by its unique identifier.
     *
     * @param id the unique identifier of the discussion to be deleted
     * @return a {@link ResponseEntity} containing no content (`Void`) and the appropriate HTTP status
     */
    ResponseEntity<Void> deleteDiscussion(String id);
    /**
     * Retrieves a paginated list of threads associated with the provided discussion identifier.
     *
     * @param discussionId the unique identifier of the discussion for which threads are to be fetched
     * @param pageable the wanted number
     * @return a paginated result of ThreadDTO objects representing the threads associated with the specified discussion
     */
    Page<ThreadDTO> getThreadsByDiscussionId(String discussionId, Pageable pageable);
    /**
     * Adds a new thread to an existing discussion.
     *
     * @param discussionId The unique identifier of the discussion to which the thread is being added.
     * @param threadDTO The details of the thread being added, represented as a {@code ThreadDTO}.
     * @return A {@code ResponseEntity<Void>} indicating the outcome of the operation. Typically, an HTTP status code
     *         is used to convey success or failure.
     */
    ResponseEntity<ThreadDTO> addThread(String discussionId, ThreadDTO threadDTO);
    /**
     * Deletes a specific thread by its unique identifier within a given discussion.
     *
     * @param discussionId the unique identifier of the discussion containing the thread
     * @param threadId the unique identifier of the thread to be deleted
     * @return a ResponseEntity with an HTTP status indicating the result of the operation
     */
    ResponseEntity<Void> deleteThread(String discussionId, String threadId);
    /**
     * Adds a reaction to a specified discussion by a user.
     *
     * @param discussionId the unique identifier of the discussion where the reaction is to be added
     * @param userId the unique identifier of the user adding the reaction to the discussion
     * @param reactionType the type of reaction to add
     * @return a ResponseEntity indicating the result of the operation
     */
    ResponseEntity<Void> addReactionToDiscussion(String discussionId, String userId, ReactionType reactionType);
    /**
     * Removes a reaction from a discussion by a specific user.
     *
     * @param discussionId the unique identifier of the discussion from which the reaction is to be removed
     * @param userId the identifier of the user whose reaction is to be removed
     * @return a ResponseEntity indicating the result of the operation
     */
    ResponseEntity<Void> removeReactionFromDiscussion(String discussionId, String userId);
    /**
     * Adds a reaction to a specific thread within a discussion.
     *
     * @param discussionId the unique identifier of the discussion containing the thread
     * @param threadId the unique identifier of the thread to which the reaction is added
     * @param userId the unique identifier of the user adding the reaction
     * @param reactionType the type of reaction to add
     * @return a ResponseEntity containing no content if the addition is successful
     */
    ResponseEntity<Void> addReactionToThread(String discussionId, String threadId, String userId, ReactionType reactionType);
    /**
     * Removes a user's reaction from a specific thread within a discussion.
     *
     * @param discussionId the unique identifier of the discussion containing the thread
     * @param threadId the unique identifier of the thread from which the reaction should be removed
     * @param userId the unique identifier of the user whose reaction will be removed from the thread
     * @return a ResponseEntity with a void body and HTTP status indicating the outcome of the operation
     */
    ResponseEntity<Void> removeReactionFromThread(String discussionId, String threadId, String userId);
    /**
     * Updates a specific thread within a discussion.
     *
     * @param discussionId the unique identifier of the discussion containing the thread
     * @param threadId the unique identifier of the thread to update
     * @param threadDTO the thread data to update (e.g., content)
     * @return a ResponseEntity indicating the result of the operation
     */
    ResponseEntity<ThreadDTO> updateThread(String discussionId, String threadId, ThreadDTO threadDTO);

    /**
     * Retrieves a paginated list of discussion data transfer objects (DTOs).
     *
     * @param pageable the page required
     * @param filterRequest the filters to be applied (optional)
     * @param includeThreads whether to include threads in the response (default: true)
     * @return a paginated result of {@link DiscussionDTO} containing the discussion data
     */
    Page<DiscussionDTO> getDiscussions(Pageable pageable, DiscussionFilterRequest filterRequest, boolean includeThreads);

    /**
     * Updates a discussion by its unique identifier.
     * Performs a partial update on the discussion entity.
     * Only provided fields in the input DTO will be updated.
     * Validates that the user is the owner of the discussion and that the resourceType is "social".
     *
     * @param discussionId the unique identifier of the discussion to update
     * @param inputUpdateDiscussionDto the data transfer object containing the fields to update
     * @param includeThreads whether to include threads in the response (default: true)
     * @param currentUser the preferredUsername of the user making the update
     * @return the updated {@link DiscussionDTO}
     */
    DiscussionDTO updateDiscussion(UUID discussionId, InputUpdateDiscussionDto inputUpdateDiscussionDto, boolean includeThreads, String currentUser);
}
