package com.finconsgroup.itserr.marketplace.event.bs.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.dto.PageRequestDto;
import com.finconsgroup.itserr.marketplace.core.web.enums.SortDirection;
import com.finconsgroup.itserr.marketplace.event.bs.bean.EventApplicationEvent;
import com.finconsgroup.itserr.marketplace.event.bs.bean.EventSubscriptionApplicationEvent;
import com.finconsgroup.itserr.marketplace.event.bs.client.InstitutionalPageDmClient;
import com.finconsgroup.itserr.marketplace.event.bs.client.UserProfileDmClient;
import com.finconsgroup.itserr.marketplace.event.bs.client.dto.institutionalpage.InputSearchInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.event.bs.client.dto.institutionalpage.OutputInstitutionalPageDmDto;
import com.finconsgroup.itserr.marketplace.event.bs.client.dto.userprofile.InputFindUserProfilesByIdsDto;
import com.finconsgroup.itserr.marketplace.event.bs.client.dto.userprofile.OutputUserProfileDmDto;
import com.finconsgroup.itserr.marketplace.event.bs.dto.InputCreateEventDto;
import com.finconsgroup.itserr.marketplace.event.bs.dto.InputProgramSubscribedParticipantDto;
import com.finconsgroup.itserr.marketplace.event.bs.dto.InputUpdateEventDto;
import com.finconsgroup.itserr.marketplace.event.bs.dto.OutputEventDto;
import com.finconsgroup.itserr.marketplace.event.bs.dto.OutputSubscribedParticipantDto;
import com.finconsgroup.itserr.marketplace.event.bs.dto.OutputUserProfileDto;
import com.finconsgroup.itserr.marketplace.event.bs.enums.MessagingEventType;
import com.finconsgroup.itserr.marketplace.event.bs.mapper.EventMapper;
import com.finconsgroup.itserr.marketplace.event.bs.service.EventService;
import com.finconsgroup.itserr.marketplace.event.dm.client.EventDmClient;
import com.finconsgroup.itserr.marketplace.event.dm.dto.OutputInstitutionalPageDto;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * Default implementation of {@link EventService} to perform operations related to event resources
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultEventService implements EventService {

    private final EventDmClient eventDmClient;
    private final EventMapper eventMapper;
    private final UserProfileDmClient userProfileDmClient;
    private final InstitutionalPageDmClient institutionalPageDmClient;
    private final ApplicationEventPublisher applicationEventPublisher;

    @NonNull
    @Override
    public OutputPageDto<OutputEventDto> findAll(Set<String> associationsToLoad, @NonNull PageRequestDto pageRequest) {
        OutputPageDto<com.finconsgroup.itserr.marketplace.event.dm.dto.OutputEventDto> outputEventPageDmDto =
                eventDmClient.findAll(associationsToLoad, pageRequest.getPageNumber(), pageRequest.getPageSize(),
                        pageRequest.getSort(), pageRequest.getDirection());

        return OutputPageDto.<OutputEventDto>builder()
                .content(mapEventDmDtoListToEventBsDtoList(outputEventPageDmDto.getContent()))
                .page(outputEventPageDmDto.getPage())
                .build();
    }

    @NonNull
    @Override
    public OutputEventDto findById(@NonNull UUID eventId) {
        com.finconsgroup.itserr.marketplace.event.dm.dto.OutputEventDto outputEventDmDto =
                eventDmClient.findById(eventId);
        return mapEventDmDtoListToEventBsDtoList(List.of(outputEventDmDto)).getFirst();
    }

    @NonNull
    @Override
    public OutputEventDto create(@NonNull InputCreateEventDto inputCreateEventDto) {
        com.finconsgroup.itserr.marketplace.event.dm.dto.OutputEventDto outputEventDmDto =
                eventDmClient.createEvent(inputCreateEventDto);
        OutputEventDto outputEventDto = mapEventDmDtoListToEventBsDtoList(List.of(outputEventDmDto)).getFirst();
        applicationEventPublisher.publishEvent(new EventApplicationEvent(outputEventDto, MessagingEventType.CREATED));
        return outputEventDto;
    }

    @NonNull
    @Override
    public OutputEventDto updateById(@NonNull UUID eventId, @NonNull InputUpdateEventDto inputUpdateEventDto) {
        com.finconsgroup.itserr.marketplace.event.dm.dto.OutputEventDto outputEventDmDto =
                eventDmClient.updateEventById(eventId, inputUpdateEventDto);
        OutputEventDto outputEventDto = mapEventDmDtoListToEventBsDtoList(List.of(outputEventDmDto)).getFirst();
        applicationEventPublisher.publishEvent(new EventApplicationEvent(outputEventDto, MessagingEventType.UPDATED));
        return outputEventDto;
    }

    @NonNull
    @Override
    public void deleteById(@NonNull UUID eventId) {
        OutputEventDto outputEventDto = mapEventDmDtoListToEventBsDtoList(List.of(eventDmClient.findById(eventId))).getFirst();
        eventDmClient.deleteEventById(eventId);
        applicationEventPublisher.publishEvent(new EventApplicationEvent(outputEventDto, MessagingEventType.DELETED));
    }

    @NonNull
    @Override
    public OutputEventDto register(@NonNull UUID eventId) {
        com.finconsgroup.itserr.marketplace.event.dm.dto.OutputEventDto outputEventDmDto = eventDmClient.register(eventId);
        OutputEventDto outputEventDto = mapEventDmDtoListToEventBsDtoList(List.of(outputEventDmDto)).getFirst();
        applicationEventPublisher.publishEvent(new EventSubscriptionApplicationEvent(outputEventDto, MessagingEventType.REGISTER));
        return outputEventDto;
    }

    @NonNull
    @Override
    public OutputEventDto unregister(@NonNull UUID eventId) {
        com.finconsgroup.itserr.marketplace.event.dm.dto.OutputEventDto outputEventDmDto = eventDmClient.unregister(eventId);
        OutputEventDto outputEventDto = mapEventDmDtoListToEventBsDtoList(List.of(outputEventDmDto)).getFirst();
        applicationEventPublisher.publishEvent(new EventSubscriptionApplicationEvent(outputEventDto, MessagingEventType.UNREGISTER));
        return outputEventDto;
    }

    @NonNull
    @Override
    public OutputEventDto registerProgram(@NonNull UUID eventId, @NonNull UUID programId, @NonNull InputProgramSubscribedParticipantDto inputProgramSubscribedParticipantDto) {
        com.finconsgroup.itserr.marketplace.event.dm.dto.OutputEventDto outputEventDmDto = eventDmClient.registerProgram(eventId, programId, inputProgramSubscribedParticipantDto);
        return mapEventDmDtoListToEventBsDtoList(List.of(outputEventDmDto)).getFirst();
    }

    @NonNull
    @Override
    public OutputEventDto unregisterProgram(@NonNull UUID eventId, @NonNull UUID programId) {
        com.finconsgroup.itserr.marketplace.event.dm.dto.OutputEventDto outputEventDmDto = eventDmClient.unregisterProgram(eventId, programId);
        return mapEventDmDtoListToEventBsDtoList(List.of(outputEventDmDto)).getFirst();
    }

    @NonNull
    private List<OutputEventDto> mapEventDmDtoListToEventBsDtoList(
            List<com.finconsgroup.itserr.marketplace.event.dm.dto.OutputEventDto> outputEventDmDtoList) {

        Set<UUID> institutionalPageIds = new HashSet<>();
        Set<UUID> userProfileIds = new HashSet<>();

        outputEventDmDtoList.forEach(outputEventDto -> {
            institutionalPageIds.addAll(
                    Optional.ofNullable(outputEventDto.getInstitutionalPages())
                            .orElse(List.of())
                            .stream()
                            .map(OutputInstitutionalPageDto::getId).toList());
            Optional.ofNullable(outputEventDto.getEventPlannerId()).ifPresent(userProfileIds::add);
            Optional.ofNullable(outputEventDto.getMaintainerId()).ifPresent(userProfileIds::add);
            if (outputEventDto.getSubscribedParticipants() != null) {
                outputEventDto.getSubscribedParticipants().stream()
                        .map(com.finconsgroup.itserr.marketplace.event.dm.dto.OutputSubscribedParticipantDto::getUserId)
                        .forEach(userProfileIds::add);
            }
        });

        Map<UUID, OutputUserProfileDmDto> userProfileDmDtoById = loadUserProfiles(userProfileIds);
        Map<UUID, OutputInstitutionalPageDmDto> institutionalPageDmDtoById = loadInstitutionalPages(institutionalPageIds);

        return outputEventDmDtoList.stream()
                .map(dto -> this.mapToOutputEventBsDto(dto, userProfileDmDtoById, institutionalPageDmDtoById))
                .collect(Collectors.toList());
    }

    private OutputEventDto mapToOutputEventBsDto(
            com.finconsgroup.itserr.marketplace.event.dm.dto.OutputEventDto outputEventDmDto,
            Map<UUID, OutputUserProfileDmDto> userProfileDmDtoById,
            Map<UUID, OutputInstitutionalPageDmDto> institutionalPageDmDtoById) {

        OutputEventDto outputEventDto = eventMapper.eventDmDtoToBsDto(outputEventDmDto);

        List<OutputInstitutionalPageDto> institutionalPages =
                Optional.ofNullable(outputEventDto.getInstitutionalPages()).orElse(List.of())
                        .stream()
                        .map(institutionalPage -> {
                            OutputInstitutionalPageDmDto institutionalPageDmDto = institutionalPageDmDtoById.get(institutionalPage.getId());
                            if (institutionalPageDmDto == null) {
                                // return the object as is, if not found
                                return institutionalPage;
                            } else {
                                return OutputInstitutionalPageDto.builder()
                                        .id(institutionalPage.getId())
                                        .name(institutionalPageDmDto.getName())
                                        .build();
                            }
                        })
                        .toList();
        outputEventDto.setInstitutionalPages(institutionalPages);

        outputEventDto.setEventPlanner(mapToOutputUserProfileDmDto(outputEventDto.getEventPlannerId(), userProfileDmDtoById));
        outputEventDto.setMaintainer(mapToOutputUserProfileDmDto(outputEventDto.getMaintainerId(), userProfileDmDtoById));
        if (outputEventDto.getSubscribedParticipants() != null) {
            // update the list in place to replace subscribed participant dm dto with bs dto
            for (int i = 0; i < outputEventDto.getSubscribedParticipants().size(); i++) {
                com.finconsgroup.itserr.marketplace.event.dm.dto.OutputSubscribedParticipantDto subscribedParticipantDmDto =
                        outputEventDto.getSubscribedParticipants().get(i);

                OutputSubscribedParticipantDto outputSubscribedParticipantDto =
                        eventMapper.subscribedParticipantDmDtoToBsDto(subscribedParticipantDmDto,
                                userProfileDmDtoById.get(subscribedParticipantDmDto.getUserId()));

                outputEventDto.getSubscribedParticipants().set(i, outputSubscribedParticipantDto);
            }
        }
        return outputEventDto;
    }

    private OutputUserProfileDto mapToOutputUserProfileDmDto(UUID userProfileId,
                                                             @NonNull Map<UUID, OutputUserProfileDmDto> userProfileDmDtoById) {
        if (userProfileId == null) {
            return null;
        }

        OutputUserProfileDmDto userProfileDmDto = userProfileDmDtoById.get(userProfileId);
        if (userProfileDmDto == null) {
            // return the object with just id, if not found
            return OutputUserProfileDto.builder().id(userProfileId).build();
        } else {
            return eventMapper.userProfileDmDtoToBsDto(userProfileDmDto);
        }
    }

    private Map<UUID, OutputUserProfileDmDto> loadUserProfiles(Set<UUID> userProfileIds) {
        if (userProfileIds.isEmpty()) {
            return Map.of();
        }

        Map<UUID, OutputUserProfileDmDto> userProfileDmDtoById = new HashMap<>();
        try {
            InputFindUserProfilesByIdsDto inputFindUserProfilesByIdsDto = InputFindUserProfilesByIdsDto.builder()
                    .ids(new ArrayList<>(userProfileIds))
                    .build();
            OutputPageDto<OutputUserProfileDmDto> userOutputProfileDmPageDto = userProfileDmClient.findAllByIds(
                    inputFindUserProfilesByIdsDto, 0, userProfileIds.size(), "id", SortDirection.ASC);
            userOutputProfileDmPageDto.getContent().forEach(outputUserProfileDmDto ->
                    userProfileDmDtoById.put(outputUserProfileDmDto.getId(), outputUserProfileDmDto));
        } catch (Exception e) {
            log.error("Error loading user profiles", e);
        }
        return userProfileDmDtoById;
    }

    private Map<UUID, OutputInstitutionalPageDmDto> loadInstitutionalPages(Set<UUID> institutionalPageIds) {
        if (institutionalPageIds.isEmpty()) {
            return Map.of();
        }

        Map<UUID, OutputInstitutionalPageDmDto> institutionalPageDmDtoById = new HashMap<>();
        try {
            InputSearchInstitutionalPageDto inputSearchInstitutionalPageDto = InputSearchInstitutionalPageDto.builder()
                    .ids(new ArrayList<>(institutionalPageIds))
                    .associationsToLoad(Set.of())
                    .build();
            OutputPageDto<OutputInstitutionalPageDmDto> institutionalPageDmPageDto = institutionalPageDmClient.search(
                    inputSearchInstitutionalPageDto, 0, institutionalPageIds.size(), "id", SortDirection.ASC);
            institutionalPageDmPageDto.getContent().forEach(outputUserProfileDmDto ->
                    institutionalPageDmDtoById.put(outputUserProfileDmDto.getId(), outputUserProfileDmDto));
        } catch (Exception e) {
            log.error("Error loading institutional pages", e);
        }
        return institutionalPageDmDtoById;
    }

}
