package com.finconsgroup.itserr.marketplace.event.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.JoinColumn;
import jakarta.persistence.ManyToOne;
import jakarta.persistence.PreUpdate;
import jakarta.persistence.Table;
import jakarta.persistence.Version;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.experimental.SuperBuilder;

import java.time.Instant;
import java.util.UUID;

/**
 * Entity class representing a program subscribed participant in the event system.
 */
@Entity
@Table(name = "program_subscribed_participant")
@SuperBuilder
@Getter
@Setter
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class ProgramSubscribedParticipantEntity extends AbstractUUIDEntity {

    /**
     * Association to the related program
     */
    @ManyToOne
    @JoinColumn(name = "program_id", updatable = false)
    private ProgramEntity program;

    /**
     * The ID of the related program.
     */
    @Column(name = "program_id", insertable = false, updatable = false)
    private UUID programId;

    /**
     * The ID of the subscriber user.
     */
    @Column(name = "user_id", updatable = false)
    private UUID userId;

    /**
     * True, if the participant is attending the program remotely.
     */
    @Column(name = "remote_participation", updatable = false)
    private Boolean remoteParticipation;

    /**
     * The timestamp when the user subscribed to the program was created.
     */
    @Column(name = "creation_time", nullable = false, updatable = false)
    private Instant creationTime;

    /**
     * The timestamp when the program was last updated.
     */
    @Column(name = "update_time", nullable = false)
    private Instant updateTime;

    /**
     * The version field used for optimistic locking.
     * <p>
     * This value is automatically managed by JPA to detect concurrent updates.
     * Each time the entity is updated, the version is incremented.
     * If two transactions try to update the same entity simultaneously,
     * JPA will detect the conflict based on this version
     * and throw an {@link jakarta.persistence.OptimisticLockException}.
     */
    @Version
    @Column(name = "version", nullable = false)
    private long version;

    /**
     * The ordering column.
     */
    @Column(name = "participant_order", nullable = false)
    private long participantOrder;

    @Override
    public void prePersist() {
        super.prePersist();
        Instant now = Instant.now();
        if (creationTime == null) {
            creationTime = now;
        }
        if (updateTime == null) {
            updateTime = now;
        }
    }

    /**
     * Updates {@code updateTime} just before the entity is updated.
     * <p>
     * The timestamp is stored in UTC to ensure consistent and timezone-safe timestamps.
     * The result is truncated to microseconds to match PostgreSQL's default precision
     * for {@code TIMESTAMPTZ} columns (6 digits).
     * Avoids using Hibernate's {@code @CreationTimestamp} to ensure timestamps
     * are immediately available after {@code JpaRepository.save()},
     * without requiring an explicit {@code JpaRepository.flush()}.
     * </p>
     */
    @PreUpdate
    public void onUpdate() {
        updateTime = Instant.now();
    }
}
