package com.finconsgroup.itserr.marketplace.event.dm.service;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2DuplicateResourceException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.event.dm.dto.InputCreateEventDto;
import com.finconsgroup.itserr.marketplace.event.dm.dto.InputUpdateEventDto;
import com.finconsgroup.itserr.marketplace.event.dm.dto.OutputEventDto;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;

import java.util.Set;
import java.util.UUID;

/**
 * Service for handling business logic related to Event. This interface provides methods to retrieve, create, update, and delete Event,
 *
 * <p>Example usage:
 * <pre>
 * Page<OutputEventDto> Event = EventService.findAll(pageable);
 * </pre>
 * </p>
 */
public interface EventService {

    /**
     * Retrieves a paginated list of all Event.
     *
     * @param associationsToLoad the associations to be returned on the response
     * @param pageable           the pagination information
     * @return a page of OutputEventDto representing the Event
     */
    @NonNull
    Page<OutputEventDto> findAll(Set<String> associationsToLoad, @NonNull Pageable pageable);

    /**
     * Retrieves event by id.
     *
     * @param userId  the id of the user
     * @param eventId the id of the event
     * @return an OutputEventDto representing the event
     * @throws WP2ResourceNotFoundException if no event is found with the given id
     */
    @NonNull
    OutputEventDto findById(UUID userId, @NonNull UUID eventId);

    /**
     * Creates new event for user using the provided input data.
     *
     * @param userId              the id of the user
     * @param inputCreateEventDto the input DTO containing event details
     */
    @NonNull
    OutputEventDto create(@NonNull UUID userId, @NonNull InputCreateEventDto inputCreateEventDto);

    /**
     * Updates existing event by id with the provided data.
     *
     * @param userId              the id of the user
     * @param eventId             the id of the event to update
     * @param inputUpdateEventDto the input DTO containing updated  event details
     * @return an OutputEventDto eventDto representing the updated event
     * @throws WP2ResourceNotFoundException if no event is found with the given id
     */
    @NonNull
    OutputEventDto updateById(@NonNull UUID userId,
                              @NonNull UUID eventId,
                              @NonNull InputUpdateEventDto inputUpdateEventDto);


    /**
     * Deletes event by id. It will move the event entity on a separated table containing deleted records.
     *
     * @param userId  the id of the user
     * @param eventId the id of the event to delete
     * @throws WP2ResourceNotFoundException if no event is found with the given id
     */
    @NonNull
    void deleteById(@NonNull UUID userId, @NonNull UUID eventId);

    /**
     * register a user to an event
     *
     * @param eventId the id of the event
     * @param userId  the id of the user
     * @throws WP2ResourceNotFoundException  if no event is found with the given id
     * @throws WP2DuplicateResourceException if user is already subscribed to the event
     */
    @NonNull
    OutputEventDto register(@NonNull UUID userId, @NonNull UUID eventId);

    /**
     * unregister a user to an event
     *
     * @param eventId the id of the event
     * @param userId  the id of the user
     * @throws WP2ResourceNotFoundException if no event is found with the given id
     */
    @NonNull
    OutputEventDto unregister(@NonNull UUID userId, @NonNull UUID eventId);

    /**
     * Retrieves event by id.
     *
     * @param eventId the id of the event
     * @return an OutputEventDto representing the event
     * @throws WP2ResourceNotFoundException if no event is found with the given id
     */
    @NonNull
    OutputEventDto getById(@NonNull UUID eventId);

    /**
     * populate the participant details
     *
     * @param outputEventDto the event
     * @throws WP2ResourceNotFoundException if no event is found with the given id
     */
    @NonNull
    void extractSubscribedParticipantProgramDetails(@NonNull OutputEventDto outputEventDto);
}
