package com.finconsgroup.itserr.marketplace.event.dm.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2DuplicateResourceException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.event.dm.dto.InputCreateEventDto;
import com.finconsgroup.itserr.marketplace.event.dm.dto.InputEventConductorDto;
import com.finconsgroup.itserr.marketplace.event.dm.dto.InputProgramDto;
import com.finconsgroup.itserr.marketplace.event.dm.dto.InputScheduleDto;
import com.finconsgroup.itserr.marketplace.event.dm.dto.InputUpdateEventDto;
import com.finconsgroup.itserr.marketplace.event.dm.dto.OutputEventConductorDto;
import com.finconsgroup.itserr.marketplace.event.dm.dto.OutputEventDto;
import com.finconsgroup.itserr.marketplace.event.dm.dto.OutputProgramDto;
import com.finconsgroup.itserr.marketplace.event.dm.dto.OutputScheduleDto;
import com.finconsgroup.itserr.marketplace.event.dm.dto.OutputSubscribedParticipantDto;
import com.finconsgroup.itserr.marketplace.event.dm.dto.OutputSubscribedProgramDto;
import com.finconsgroup.itserr.marketplace.event.dm.entity.ArchivedEventEntity;
import com.finconsgroup.itserr.marketplace.event.dm.entity.ArchivedSubscribedParticipantEntity;
import com.finconsgroup.itserr.marketplace.event.dm.entity.EventConductorEntity;
import com.finconsgroup.itserr.marketplace.event.dm.entity.EventEntity;
import com.finconsgroup.itserr.marketplace.event.dm.entity.ProgramConductorEntity;
import com.finconsgroup.itserr.marketplace.event.dm.entity.ProgramEntity;
import com.finconsgroup.itserr.marketplace.event.dm.entity.ScheduleEntity;
import com.finconsgroup.itserr.marketplace.event.dm.entity.SubscribedParticipantEntity;
import com.finconsgroup.itserr.marketplace.event.dm.mapper.ArchivedEventMapper;
import com.finconsgroup.itserr.marketplace.event.dm.mapper.ArchivedSubscribedParticipantMapper;
import com.finconsgroup.itserr.marketplace.event.dm.mapper.EventConductorMapper;
import com.finconsgroup.itserr.marketplace.event.dm.mapper.EventMapper;
import com.finconsgroup.itserr.marketplace.event.dm.mapper.ProgramMapper;
import com.finconsgroup.itserr.marketplace.event.dm.mapper.ScheduleMapper;
import com.finconsgroup.itserr.marketplace.event.dm.mapper.SubscribedParticipantMapper;
import com.finconsgroup.itserr.marketplace.event.dm.repository.ArchivedEventRepository;
import com.finconsgroup.itserr.marketplace.event.dm.repository.ArchivedSubscribedParticipantRepository;
import com.finconsgroup.itserr.marketplace.event.dm.repository.EventConductorRepository;
import com.finconsgroup.itserr.marketplace.event.dm.repository.EventRepository;
import com.finconsgroup.itserr.marketplace.event.dm.repository.ProgramRepository;
import com.finconsgroup.itserr.marketplace.event.dm.repository.ScheduleRepository;
import com.finconsgroup.itserr.marketplace.event.dm.repository.SubscribedParticipantRepository;
import com.finconsgroup.itserr.marketplace.event.dm.service.EventService;
import com.finconsgroup.itserr.marketplace.event.dm.util.DateTimeUtils;
import lombok.Builder;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDate;
import java.time.LocalTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;

import static com.finconsgroup.itserr.marketplace.event.dm.util.StreamUtils.safeStream;

/**
 * Default implementation of {@link EventService} to perform operations related to event resources
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultEventService implements EventService {

    private static final String ASSOCIATION_ALL = "all";
    private static final String ASSOCIATION_EVENT_CONDUCTORS = "eventConductors";
    private static final String ASSOCIATION_SCHEDULES = "schedules";
    private static final String ASSOCIATION_SUBSCRIBED_PARTICIPANTS = "subscribedParticipants";
    private static final String ASSOCIATION_PROGRAMS = "programs";

    private static final String FORMAT_BUSINESS_KEY = "{userId: %s, eventId: %s}";

    private final EventRepository eventRepository;
    private final EventMapper eventMapper;
    private final EventConductorRepository eventConductorRepository;
    private final EventConductorMapper eventConductorMapper;
    private final ScheduleRepository scheduleRepository;
    private final ScheduleMapper scheduleMapper;
    private final ProgramRepository programRepository;
    private final ProgramMapper programMapper;
    private final SubscribedParticipantRepository subscribedParticipantRepository;
    private final SubscribedParticipantMapper subscribedParticipantMapper;

    private final ArchivedEventRepository archivedEventRepository;
    private final ArchivedEventMapper archivedEventMapper;

    private final ArchivedSubscribedParticipantRepository archivedSubscribedParticipantRepository;
    private final ArchivedSubscribedParticipantMapper archivedSubscribedParticipantMapper;

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputEventDto> findAll(Set<String> associationsToLoad, @NonNull Pageable pageable) {
        Page<EventEntity> eventEntityPage = eventRepository.findAll(pageable);
        return mapEntitiesToDtos(eventEntityPage, associationsToLoad);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public OutputEventDto findById(UUID userId, @NonNull UUID eventId) {
        EventEntity eventEntity = findByIdOrThrow(eventId);
        OutputEventDto outputEventDto = eventMapper.toDto(eventEntity);
        if (userId != null) {
            outputEventDto.setSubscribed(checkSubscription(outputEventDto, userId));
        }
        extractSubscribedParticipantProgramDetails(outputEventDto);
        return outputEventDto;
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputEventDto create(@NonNull UUID userId, @NonNull InputCreateEventDto inputCreateEventDto) {
        EventEntity eventEntity = eventMapper.toEntity(inputCreateEventDto);
        eventEntity.setEventPlannerId(userId);
        eventEntity.setMaintainerId(userId);
        updateAssociationsForSave(eventEntity);
        eventEntity.setSubscribedParticipantsCount(0);
        return eventMapper.toDto(eventRepository.saveAndFlush(eventEntity));
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputEventDto updateById(@NonNull UUID userId, @NonNull UUID eventId,
                                     @NonNull InputUpdateEventDto inputUpdateEventDto) {
        EventEntity eventEntity = findByUserIdAndIdOrThrow(userId, eventId);
        updateEntity(inputUpdateEventDto, eventEntity);
        updateAssociationsForSave(eventEntity);
        EventEntity savedEventEntity = eventRepository.saveAndFlush(eventEntity);
        OutputEventDto outputEventDto = eventMapper.toDto(savedEventEntity);
        outputEventDto.setSubscribed(checkSubscription(outputEventDto, userId));
        extractSubscribedParticipantProgramDetails(outputEventDto);
        return outputEventDto;
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public void deleteById(@NonNull UUID userId, @NonNull UUID eventId) {
        EventEntity eventEntity = findByUserIdAndIdOrThrow(userId, eventId);
        ArchivedEventEntity archivedEventEntity = archivedEventMapper.toArchivedEntity(eventEntity);
        archivedEventEntity.getSubscribedParticipants()
                .forEach(archivedSPEntity -> archivedSPEntity.setEvent(archivedEventEntity));
        safeStream(archivedEventEntity.getSchedules())
                .flatMap(schedule -> safeStream(schedule.getPrograms()))
                .forEach(program ->
                        safeStream(program.getSubscribedParticipants())
                                .forEach(participant -> participant.setProgram(program))
                );
        archivedEventRepository.save(archivedEventEntity);
        eventRepository.delete(eventEntity);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputEventDto register(@NonNull UUID userId, @NonNull UUID eventId) {
        EventEntity existingEventEntity = findByIdOrThrow(eventId);
        if (subscribedParticipantRepository.existsByUserIdAndEventId(userId, eventId)) {
            throw new WP2DuplicateResourceException(FORMAT_BUSINESS_KEY.formatted(userId, eventId));
        }

        SubscribedParticipantEntity subscribedParticipantEntity = SubscribedParticipantEntity
                .builder()
                .userId(userId)
                .eventId(eventId)
                .event(existingEventEntity)
                .participantOrder(existingEventEntity.getSubscribedParticipantsCount())
                .build();
        subscribedParticipantRepository.saveAndFlush(subscribedParticipantEntity);

        EventEntity eventEntity = findByIdOrThrow(eventId);
        eventEntity.incrementParticipants();
        eventRepository.saveAndFlush(eventEntity);

        OutputEventDto outputEventDto = eventMapper.toDto(eventEntity);
        extractSubscribedParticipantProgramDetails(outputEventDto);
        outputEventDto.setSubscribed(true);
        return outputEventDto;
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputEventDto unregister(@NonNull UUID userId, @NonNull UUID eventId) {
        SubscribedParticipantEntity subscribedParticipantEntity = subscribedParticipantRepository.findByUserIdAndEventId(userId, eventId)
                .orElseThrow(() -> new WP2ResourceNotFoundException(FORMAT_BUSINESS_KEY.formatted(userId, eventId)));
        subscribedParticipantRepository.delete(subscribedParticipantEntity);
        ArchivedSubscribedParticipantEntity archivedSubscribedParticipantEntity = archivedSubscribedParticipantMapper
                .toArchivedSubscribedParticipantEntity(subscribedParticipantEntity);
        archivedSubscribedParticipantEntity.setEvent(ArchivedEventEntity.builder().id(eventId).build());
        archivedSubscribedParticipantRepository.save(archivedSubscribedParticipantEntity);
        eventRepository.flush();

        EventEntity eventEntity = findByIdOrThrow(eventId);
        eventEntity.decrementParticipants();
        eventRepository.saveAndFlush(eventEntity);

        OutputEventDto outputEventDto = eventMapper.toDto(eventEntity);
        extractSubscribedParticipantProgramDetails(outputEventDto);
        outputEventDto.setSubscribed(false);
        return outputEventDto;
    }

    private EventEntity findByIdOrThrow(@NonNull UUID eventId) {
        return eventRepository
                .findById(eventId)
                .orElseThrow(() -> new WP2ResourceNotFoundException(eventId));
    }

    private Boolean checkSubscription(@NonNull OutputEventDto outputEventDto, @NonNull UUID userId) {
        return Optional.ofNullable(outputEventDto.getSubscribedParticipants())
                .orElse(List.of())
                .stream()
                .anyMatch(dto -> userId.equals(dto.getUserId()));
    }

    private EventEntity findByUserIdAndIdOrThrow(@NonNull UUID userId, @NonNull UUID eventId) {
        return eventRepository
                .findByMaintainerIdAndId(userId, eventId)
                .orElseThrow(() -> new WP2ResourceNotFoundException(eventId));
    }

    @NonNull
    public OutputEventDto getById(@NonNull UUID eventId) {
        EventEntity eventEntity = eventRepository
                .findById(eventId)
                .orElseThrow(() -> new WP2ResourceNotFoundException(eventId));
        return eventMapper.toDto(eventEntity);
    }

    /**
     * Performs load for all related associations that are needed and maps from entity to dto using the loaded associations.
     *
     * @param eventEntityPage the page to map
     * @return a page of mapped OutputInstitutionalPageDto
     */
    private Page<OutputEventDto> mapEntitiesToDtos(final Page<EventEntity> eventEntityPage,
                                                   final Set<String> associationsToLoad) {
        Set<String> associationsToLoadOrDefault = Optional.ofNullable(associationsToLoad).orElse(Set.of());
        LoadedAssociations loadedAssociations = loadAssociations(eventEntityPage, associationsToLoadOrDefault);
        return eventEntityPage.map(entity ->
                mapEntityToDtoWithAssociations(entity, associationsToLoadOrDefault, loadedAssociations));
    }

    /**
     * Performs eager load for all related associations that are needed for mapping a page on events.
     *
     * @param eventEntityPage the page containing loaded entities
     * @return the loaded associations
     */
    private LoadedAssociations loadAssociations(final Page<EventEntity> eventEntityPage,
                                                final Set<String> associationsToLoad) {
        LoadedAssociations loadedAssociations = new LoadedAssociations();

        List<UUID> loadedEventIds = eventEntityPage
                .getContent().stream().map(EventEntity::getId).toList();

        boolean loadAllAssociations = associationsToLoad.contains(ASSOCIATION_ALL);

        if (loadAllAssociations || associationsToLoad.contains(ASSOCIATION_EVENT_CONDUCTORS)) {
            // retrieve event conductors
            if (!loadedEventIds.isEmpty()) {
                eventConductorRepository.findAllByEventIdIn(loadedEventIds)
                        .forEach(entity ->
                                loadedAssociations.eventConductorsMap.computeIfAbsent(entity.getEventId(),
                                                k -> new LinkedList<>())
                                        .add(eventConductorMapper.toDto(entity)));
            }
        }

        if (loadAllAssociations || associationsToLoad.contains(ASSOCIATION_SCHEDULES) || associationsToLoad.contains(ASSOCIATION_PROGRAMS)) {
            // retrieve schedules
            if (!loadedEventIds.isEmpty()) {
                List<ScheduleEntity> scheduleEntities =
                        scheduleRepository.findAllByEventIdIn(loadedEventIds);

                if (!scheduleEntities.isEmpty()) {
                    scheduleEntities.forEach(entity ->
                            loadedAssociations.schedulesMap.computeIfAbsent(
                                            entity.getEventId(), k -> new LinkedList<>())
                                    .add(scheduleMapper.toDto(entity))
                    );

                    if (loadAllAssociations || associationsToLoad.contains(ASSOCIATION_PROGRAMS)) {
                        // retrieve programs
                        List<ProgramEntity> programEntities =
                                programRepository.findAllByScheduleEventIdIn(loadedEventIds);
                        if (!programEntities.isEmpty()) {
                            programEntities.forEach(entity ->
                                    loadedAssociations.programsMap.computeIfAbsent(
                                                    entity.getScheduleId(), k -> new LinkedList<>())
                                            .add(programMapper.toDto(entity)));
                        }
                    }
                }
            }
        }

        if (loadAllAssociations || associationsToLoad.contains(ASSOCIATION_SUBSCRIBED_PARTICIPANTS)) {
            // retrieve subscribed participant user ids
            List<SubscribedParticipantEntity> subscribedParticipantEntities =
                    subscribedParticipantRepository.findAllByEventIdIn(loadedEventIds);
            if (!subscribedParticipantEntities.isEmpty()) {
                subscribedParticipantEntities.forEach(entity ->
                        loadedAssociations.subscribedParticipantsMap.computeIfAbsent(
                                        entity.getEventId(), k -> new LinkedList<>())
                                .add(subscribedParticipantMapper.toDto(entity)));
            }
        }

        return loadedAssociations;
    }

    private OutputEventDto mapEntityToDtoWithAssociations(final EventEntity eventEntity,
                                                          final Set<String> associationsToLoad,
                                                          final LoadedAssociations loadedAssociations) {

        Associations.AssociationsBuilder associationsBuilder = Associations.builder();
        boolean loadAllAssociations = associationsToLoad.contains(ASSOCIATION_ALL);
        if (loadAllAssociations || associationsToLoad.contains(ASSOCIATION_EVENT_CONDUCTORS)) {
            associationsBuilder.eventConductors(loadedAssociations.eventConductorsMap.get(eventEntity.getId()));
        }
        if (loadAllAssociations || associationsToLoad.contains(ASSOCIATION_SCHEDULES) || associationsToLoad.contains(ASSOCIATION_PROGRAMS)) {
            List<OutputScheduleDto> scheduleDtos = loadedAssociations.schedulesMap.get(eventEntity.getId());
            if (loadAllAssociations || associationsToLoad.contains(ASSOCIATION_PROGRAMS)) {
                scheduleDtos.forEach(outputScheduleDto -> outputScheduleDto.setPrograms(loadedAssociations.programsMap.get(outputScheduleDto.getId())));
            }
            associationsBuilder.schedules(scheduleDtos);
        }
        if (loadAllAssociations || associationsToLoad.contains(ASSOCIATION_SUBSCRIBED_PARTICIPANTS)) {
            associationsBuilder.subscribedParticipants(loadedAssociations.subscribedParticipantsMap.get(eventEntity.getId()));
        }

        // map entity to dto
        @SuppressWarnings("UnnecessaryLocalVariable")
        OutputEventDto outputEventDto = eventMapper.toDtoWithAssociations(
                eventEntity, associationsBuilder.build()
        );

        return outputEventDto;
    }

    private void updateEntity(InputUpdateEventDto inputUpdateEventDto, EventEntity eventEntity) {
        eventMapper.updateEntity(inputUpdateEventDto, eventEntity);
        updateEventConductors(inputUpdateEventDto, eventEntity);
        updateSchedules(inputUpdateEventDto, eventEntity);
        updateAssociationsForSave(eventEntity);
    }

    private void updateEventConductors(InputUpdateEventDto inputUpdateEventDto, EventEntity eventEntity) {
        ListIterator<InputEventConductorDto> inputEventConductorsDtoIterator = inputUpdateEventDto.getEventConductors() != null
                ? inputUpdateEventDto.getEventConductors().listIterator()
                : Collections.emptyListIterator();
        ListIterator<EventConductorEntity> entityEventConductorsIterator = eventEntity.getEventConductors() != null
                ? eventEntity.getEventConductors().listIterator()
                : Collections.emptyListIterator();
        // update existing eventConductor entities with provided input DTOs
        while (inputEventConductorsDtoIterator.hasNext() && entityEventConductorsIterator.hasNext()) {
            InputEventConductorDto inputEventConductorDto = inputEventConductorsDtoIterator.next();
            EventConductorEntity eventConductorEntity = entityEventConductorsIterator.next();
            eventConductorMapper.updateEntity(inputEventConductorDto, eventConductorEntity);
        }
        // add new entity eventConductors, as there are more input DTOs provided
        while (inputEventConductorsDtoIterator.hasNext()) {
            InputEventConductorDto inputEventConductorDto = inputEventConductorsDtoIterator.next();
            EventConductorEntity eventConductorEntity = eventConductorMapper.toEntity(inputEventConductorDto);
            entityEventConductorsIterator.add(eventConductorEntity);
        }
        // Remove remaining entity eventConductors, as there are NO more input DTOs provided
        while (entityEventConductorsIterator.hasNext()) {
            entityEventConductorsIterator.next();
            entityEventConductorsIterator.remove();
        }
    }

    private void updateSchedules(InputUpdateEventDto inputUpdateEventDto, EventEntity eventEntity) {
        ListIterator<InputScheduleDto> inputSchedulesDtoIterator = inputUpdateEventDto.getSchedules() != null
                ? inputUpdateEventDto.getSchedules().listIterator()
                : Collections.emptyListIterator();
        ListIterator<ScheduleEntity> entitySchedulesIterator = eventEntity.getSchedules() != null
                ? eventEntity.getSchedules().listIterator()
                : Collections.emptyListIterator();
        // update existing schedule entities with provided input DTOs
        while (inputSchedulesDtoIterator.hasNext() && entitySchedulesIterator.hasNext()) {
            InputScheduleDto inputScheduleDto = inputSchedulesDtoIterator.next();
            ScheduleEntity scheduleEntity = entitySchedulesIterator.next();
            updateScheduleEntity(inputScheduleDto, scheduleEntity);
        }
        // add new entity schedules, as there are more input DTOs provided
        while (inputSchedulesDtoIterator.hasNext()) {
            InputScheduleDto inputScheduleDto = inputSchedulesDtoIterator.next();
            ScheduleEntity scheduleEntity = scheduleMapper.toEntity(inputScheduleDto);
            entitySchedulesIterator.add(scheduleEntity);
        }
        // Remove remaining entity schedules, as there are NO more input DTOs provided
        while (entitySchedulesIterator.hasNext()) {
            entitySchedulesIterator.next();
            entitySchedulesIterator.remove();
        }
    }

    private void updateScheduleEntity(InputScheduleDto inputScheduleDto, ScheduleEntity scheduleEntity) {
        scheduleMapper.updateEntity(inputScheduleDto, scheduleEntity);
        updatePrograms(inputScheduleDto, scheduleEntity);
    }

    private void updatePrograms(InputScheduleDto inputScheduleDto, ScheduleEntity scheduleEntity) {
        ListIterator<InputProgramDto> inputProgramsDtoIterator = inputScheduleDto.getPrograms() != null
                ? inputScheduleDto.getPrograms().listIterator()
                : Collections.emptyListIterator();
        ListIterator<ProgramEntity> entityProgramsIterator = scheduleEntity.getPrograms() != null
                ? scheduleEntity.getPrograms().listIterator()
                : Collections.emptyListIterator();
        // update existing program entities with provided input DTOs
        while (inputProgramsDtoIterator.hasNext() && entityProgramsIterator.hasNext()) {
            InputProgramDto inputProgramDto = inputProgramsDtoIterator.next();
            ProgramEntity programEntity = entityProgramsIterator.next();
            programMapper.updateEntity(inputProgramDto, programEntity);
        }
        // add new entity programs, as there are more input DTOs provided
        while (inputProgramsDtoIterator.hasNext()) {
            InputProgramDto inputProgramDto = inputProgramsDtoIterator.next();
            ProgramEntity programEntity = programMapper.toEntity(inputProgramDto);
            entityProgramsIterator.add(programEntity);
        }
        // Remove remaining entity programs, as there are NO more input DTOs provided
        while (entityProgramsIterator.hasNext()) {
            entityProgramsIterator.next();
            entityProgramsIterator.remove();
        }
    }

    private void updateAssociationsForSave(@NonNull EventEntity eventEntity) {
        LocalDate eventStartDate = null;
        LocalDate eventEndDate = null;
        LocalTime schdeduleStartTime = null;
        LocalTime scheduleEndTime = null;
        for (EventConductorEntity eventConductorEntity : eventEntity.getEventConductors()) {
            eventConductorEntity.setEvent(eventEntity);
        }
        for (ScheduleEntity scheduleEntity : eventEntity.getSchedules()) {
            eventStartDate = DateTimeUtils.min(eventStartDate, scheduleEntity.getStartDate());
            eventEndDate = DateTimeUtils.max(eventEndDate, scheduleEntity.getStartDate());

            if (scheduleEntity.getPrograms() != null) {
                for (ProgramEntity programEntity : scheduleEntity.getPrograms()) {
                    schdeduleStartTime = DateTimeUtils.min(schdeduleStartTime, programEntity.getEndTime());
                    scheduleEndTime = DateTimeUtils.max(scheduleEndTime, programEntity.getEndTime());

                    if (programEntity.getSubscribedParticipantsCount() == null) {
                        programEntity.setSubscribedParticipantsCount(0);
                    }

                    for (ProgramConductorEntity programConductorEntity : programEntity.getProgramConductors()) {
                        programConductorEntity.setProgram(programEntity);
                    }

                    programEntity.setSchedule(scheduleEntity);
                }

                // set calculated start/end time based on the programs
                // if none provided on the schedule by the user
                if (scheduleEntity.getStartTime() == null) {
                    scheduleEntity.setStartTime(schdeduleStartTime);
                }
                if (scheduleEntity.getEndTime() == null) {
                    scheduleEntity.setEndTime(scheduleEndTime);
                }
            }
            scheduleEntity.setEvent(eventEntity);
        }
        // set calculated start/end date based on the schedules
        eventEntity.setStartDate(eventStartDate);
        eventEntity.setEndDate(eventEndDate);
    }

    public void extractSubscribedParticipantProgramDetails(@NonNull OutputEventDto outputEventDto) {
        Map<UUID, List<OutputSubscribedProgramDto>> uuidOutputSpProgramMap = new HashMap<>();

        safeStream(outputEventDto.getSchedules())
                .flatMap(schedule -> safeStream(schedule.getPrograms()))
                .forEach(program ->
                        safeStream(program.getSubscribedParticipants())
                                .forEach(participant -> {
                                    UUID userId = participant.getUserId();
                                    OutputSubscribedProgramDto sbProgramDto = OutputSubscribedProgramDto.builder()
                                            .id(program.getId())
                                            .title(program.getTitle())
                                            .description(program.getDescription())
                                            .remoteParticipation(participant.getRemoteParticipation())
                                            .build();

                                    uuidOutputSpProgramMap
                                            .computeIfAbsent(userId, k -> new ArrayList<>())
                                            .add(sbProgramDto);
                                })
                );

        safeStream(outputEventDto.getSubscribedParticipants())
                .forEach(participant -> {
                    List<OutputSubscribedProgramDto> programs = uuidOutputSpProgramMap.get(participant.getUserId());
                    if (programs != null) {
                        participant.setSubscribedPrograms(programs);
                    }
                });
    }

    /**
     * Container class to store the associations that have been loaded for a paginated list of events.
     */
    @RequiredArgsConstructor
    private static class LoadedAssociations {
        private final Map<UUID, List<OutputEventConductorDto>> eventConductorsMap = new HashMap<>();
        private final Map<UUID, List<OutputScheduleDto>> schedulesMap = new HashMap<>();
        private final Map<UUID, List<OutputProgramDto>> programsMap = new HashMap<>();
        private final Map<UUID, List<OutputSubscribedParticipantDto>> subscribedParticipantsMap = new HashMap<>();
    }

    /**
     * Container class to store the associations that have been mapped for single event.
     */
    @Builder
    @Getter
    public static class Associations {
        private final List<OutputEventConductorDto> eventConductors;
        private final List<OutputScheduleDto> schedules;
        private final List<OutputSubscribedParticipantDto> subscribedParticipants;
    }
}
