package com.finconsgroup.itserr.marketplace.favouritesearch.dm.api;

import com.finconsgroup.itserr.marketplace.favouritesearch.dm.dto.InputCreateFavouriteSearchDto;
import com.finconsgroup.itserr.marketplace.favouritesearch.dm.dto.OutputFavouriteSearchDto;
import com.finconsgroup.itserr.marketplace.favouritesearch.dm.enums.SearchContext;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.Positive;
import jakarta.validation.constraints.PositiveOrZero;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Sort;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.UUID;

/**
 * This interface defines the contract for REST API endpoints related to favourite searches management.
 *
 * <p>
 * It provides endpoints for retrieving, creating, and deleting favourite searches.
 * </p>
 *
 * <p>Example usage:
 * <pre>
 * POST     /api/v1/dm/favourite-searches - Create a new favouriteSearch
 * GET      /api/v1/dm/favourite-searches - Retrieve a paginated list of all favouriteSearches.
 * GET      /api/v1/dm/favourite-searches/{favouriteSearchId} - Retrieve a favouriteSearch by id.
 * DELETE   /api/v1/dm/favourite-searches/{favouriteSearchId} - Delete a favouriteSearch by id.
 * </pre>
 * </p>
 */
@Tag(
        name = "FavouriteSearch",
        description = "The FavouriteSearch API: it provides endpoints for " +
                "retrieving, creating, and deleting FavouriteSearch."
)
@SecurityRequirement(name = "BearerAuth")
public interface FavouriteSearchApi {

    /**
     * Creates a new favourite search for the authenticated user.
     *
     * @param inputCreateFavouriteSearchDto the input data transfer object containing the favourite search details
     * @return the created {@link OutputFavouriteSearchDto} and HTTP status 201 (Created)
     */
    @Operation(
            summary = "create a favouriteSearch for the authenticated user",
            responses = {@ApiResponse(responseCode = "201", description = "Created")}
    )
    @PostMapping(
            value = "/favourite-searches",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.CREATED)
    OutputFavouriteSearchDto create(
            @Valid @RequestBody InputCreateFavouriteSearchDto inputCreateFavouriteSearchDto
    );

    /**
     * Retrieves a paginated list of all favourite searches for the authenticated user.
     *
     * @param context    the text to filter FavouriteSearch by context
     * @param searchText the text to filter FavouriteSearch by
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize   the number of FavouriteSearch per page (default is 10)
     * @param sort       the field to sort by (default is "creationTime")
     * @param direction  the direction of sorting (default is descending)
     * @return a page of {@link OutputFavouriteSearchDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "find all favouriteSearches for the authenticated user",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @GetMapping(value = "/favourite-searches", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    Page<OutputFavouriteSearchDto> findAll(
            @RequestParam(name = "context", required = false) SearchContext context,
            @RequestParam(name = "searchText", defaultValue = "", required = false) String searchText,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "creationTime", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "DESC", required = false) Sort.Direction direction
    );

    /**
     * Retrieves a favourite search by id for the authenticated user.
     *
     * @param favouriteSearchId the id of the favouriteSearch to retrieve
     * @return the found {@link OutputFavouriteSearchDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "find favouriteSearch by id for the authenticated user",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found"),
            }
    )
    @GetMapping(value = "/favourite-searches/{favouriteSearchId}", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    OutputFavouriteSearchDto findById(
            @PathVariable("favouriteSearchId") UUID favouriteSearchId
    );

    /**
     * Deletes a favourite search by id for the authenticated user.
     *
     * @param favouriteSearchId the id of the favouriteSearch to delete
     */
    @Operation(
            summary = "delete favouriteSearch by id for the authenticated user",
            responses = {
                    @ApiResponse(responseCode = "204", description = "No Content"),
                    @ApiResponse(responseCode = "404", description = "Not Found"),
            }
    )
    @ResponseStatus(HttpStatus.NO_CONTENT)
    @DeleteMapping(value = "/favourite-searches/{favouriteSearchId}", produces = MediaType.APPLICATION_JSON_VALUE)
    void deleteById(
            @PathVariable("favouriteSearchId") UUID favouriteSearchId
    );

}
