/**
 * Global exception handler for REST controllers in the searchhistory system.
 * This class handles custom exceptions related to searchHistoryEntry and category operations
 * and returns appropriate HTTP error responses.
 *
 * <p>Example usage:
 * <pre>
 * When a FavouriteSearchNotFoundException is thrown, this handler returns a 404 NOT FOUND response.
 * </pre>
 * </p>
 */
package com.finconsgroup.itserr.marketplace.favouritesearch.dm.exception;

import com.finconsgroup.itserr.marketplace.core.web.exception.ErrorResponseDto;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.context.request.WebRequest;

import static com.finconsgroup.itserr.marketplace.core.web.exception.ExceptionUtils.createResponseEntityError;
import static com.finconsgroup.itserr.marketplace.core.web.exception.ExceptionUtils.logError;
import static org.springframework.http.HttpStatus.NOT_FOUND;

/**
 * Custom Exception Handler.
 */
@RestControllerAdvice
public class CustomExceptionHandler {

    /**
     * Handles FavouriteSearchNotFoundException and returns a 404 NOT FOUND error response.
     *
     * @param ex      the FavouriteSearchNotFoundException that was thrown
     * @param request the current web request
     * @return a ResponseEntity containing the error response DTO and HTTP status 404
     */
    @ExceptionHandler({FavouriteSearchNotFoundException.class})
    public ResponseEntity<ErrorResponseDto> handleFavouriteSearchNotFoundException(
            final FavouriteSearchNotFoundException ex, final WebRequest request
    ) {
        logError(ex, request);
        return createResponseEntityError(NOT_FOUND, ex.getMessage());
    }

}
