package com.finconsgroup.itserr.marketplace.favourite.user.bs.mapper;

import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.OutputFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.messaging.dto.FavouriteItemMessageBodyDto;
import com.finconsgroup.itserr.messaging.dto.MessagingEventUserDto;
import org.mapstruct.Mapper;
import org.mapstruct.Mapping;

import java.time.Instant;
import java.util.List;

/**
 * Mapper interface for converting favourite item-related objects between different representations.
 */
@Mapper(config = MapperConfiguration.class)
public interface FavouriteItemMapper {

    /**
     * Converts favourite item data to a message body DTO.
     *
     * @param src The source favourite item DTO
     * @param name The name of the item
     * @param title The title of the item
     * @param notifyUserIds The users a notification should be sent to
     * @param instant The timestamp of the conversion
     * @return The converted message body DTO
     */
    @Mapping(target = "category", source = "src.context")
    @Mapping(target = "name", ignore = true)
    @Mapping(target = "title", ignore = true)
    @Mapping(target = "message", ignore = true)
    @Mapping(target = "status", ignore = true)
    @Mapping(target = "timestamp", source = "instant")
    @Mapping(target = "user", expression = "java(this.getMessagingEventUserFromSecurity())")
    @Mapping(target = "additionalData", expression = "java(this.toFavouriteItemMessageBodyDtoAdditionalData(src, name, title, notifyUserIds))")
    FavouriteItemMessageBodyDto toFavouriteItemMessageBodyDto(
            OutputFavouriteUserItemDto src,
            String name,
            String title,
            List<String> notifyUserIds,
            Instant instant);

    /**
     * Creates additional data for the favourite item message DTO.
     *
     * @param src The source favourite item DTO
     * @param notifyUserIds Users a notification should be sent to
     * @return Additional data object for the message
     */
    FavouriteItemMessageBodyDto.AdditionalData toFavouriteItemMessageBodyDtoAdditionalData(
            OutputFavouriteUserItemDto src,
            String itemName,
            String itemTitle,
            List<String> notifyUserIds);

    /**
     * Retrieves user information from the security context for messaging events.
     *
     * @return A DTO containing the current user's information
     */
    default MessagingEventUserDto getMessagingEventUserFromSecurity() {
        return MessagingEventUserDto.builder()
                .id(JwtTokenHolder.getUserId().orElse(null))
                .username(JwtTokenHolder.getPreferredUsername().orElse(null))
                .name(JwtTokenHolder.getName().orElse(""))
                .build();
    }

}
