package com.finconsgroup.itserr.marketplace.favourite.user.bs.messaging.helper;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.client.CatalogBsClient;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.OutputFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.catalog.OutputItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.mapper.FavouriteItemMapper;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.messaging.dto.CatalogItemMessagingStatusAdditionalDataDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.messaging.dto.FavouriteItemMessageBodyDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.messaging.dto.InstitutionalPageStatusChangeNotificationData;
import io.micrometer.common.util.StringUtils;
import lombok.Builder;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

import java.time.Instant;
import java.util.List;

/**
 * Helper class for building messaging-related DTOs and handling messaging operations.
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class MessagingHelper {

    /** Client for interacting with the Catalog service. */
    private final CatalogBsClient catalogBsClient;

    /** Mapper for converting favourite item */
    private final FavouriteItemMapper favouriteItemMapper;

    private final ObjectMapper objectMapper;

    /**
     * Favourite referred item data.
     *
     * @param name Name.
     * @param title Title.
     */
    @Builder
    private record ItemData(
            String name,
            String title,
            String owner ) {
    }

    /**
     * Builds a message DTO for a newly favourite item.
     *
     * @param favouriteItem The favourite item
     * @return A DTO containing the favourite item details and related information
     */
    public FavouriteItemMessageBodyDto buildFavouriteItemMessage(
            final OutputFavouriteUserItemDto favouriteItem) {

        final ItemData data = getData(favouriteItem);

        return favouriteItemMapper.toFavouriteItemMessageBodyDto(
                favouriteItem,
                data.name(),
                data.title(),
                data.owner() != null ? List.of(data.owner()) : List.of(),
                Instant.now());
    }

    public CatalogItemMessagingStatusAdditionalDataDto convertToCatalogItemMessagingStatusAdditionalDataDto(Object additionalData) {
        return convertData(additionalData, CatalogItemMessagingStatusAdditionalDataDto.class,
                "Failed to convert additional data to CatalogItemMessagingStatusAdditionalDataDto");
    }

    public InstitutionalPageStatusChangeNotificationData convertToInstitutionalPageStatusChangeNotificationData(Object additionalData) {
        return convertData(additionalData, InstitutionalPageStatusChangeNotificationData.class,
                "Failed to convert additional data to InstitutionalPageStatusChangeNotificationData");
    }

    @NonNull
    private ItemData getData(final OutputFavouriteUserItemDto favouriteItem) {

        final ItemData data = switch (favouriteItem.getContext()) {
            case CATALOG -> getCatalogItemData(favouriteItem.getItemId());
            case null, default -> null;
        };

        return data != null
                ? data
                : ItemData.builder().build();

    }

    private ItemData getCatalogItemData(final String itemId) {

        if (StringUtils.isBlank(itemId)) {
            return null;
        }

        try {
            final OutputItemDto item = catalogBsClient.getItemById(itemId);
            return ItemData.builder()
                    .title(item.getTitle())
                    .owner(item.getMaintainer().getEmail())
                    .build();
        } catch (final Exception e) {
            log.error("Error retrieving catalog item name, fallback to id", e);
        }

        // Fallback to null
        return null;

    }

    private <T> T convertData(Object additionalData, Class<T> targetType, String errorMessage) {
        try {
            return objectMapper.convertValue(additionalData, targetType);
        } catch (IllegalArgumentException e) {
            log.error("{}: {}", errorMessage, additionalData, e);
            throw new IllegalArgumentException("Invalid additional data format", e);
        }
    }

}
