package com.finconsgroup.itserr.marketplace.favourite.user.bs.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.bean.FilterProperties;
import com.finconsgroup.itserr.marketplace.core.web.utils.FilterUtils;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.bean.DetailRequest;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.config.FavouriteUserBsProperties;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.FavouriteUserItemDetail;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.enums.ItemContext;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.service.FavouriteUserItemDetailProvider;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.service.FavouriteUserItemDetailProviderRegistry;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.NonNull;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

/**
 * An abstract class implementing {@link FavouriteUserItemDetailProvider} that should be extended by
 * each implementation
 *
 * @param <T> the type of favourite user item detail result
 */
@Slf4j
public abstract class AbstractFavouriteUserItemDetailProvider<T extends FavouriteUserItemDetail> implements FavouriteUserItemDetailProvider<T> {
    protected final FavouriteUserBsProperties.ContextProperties contextProperties;
    private final FilterProperties filterProperties;

    /**
     * It registers the detail provider in the registry
     *
     * @param favouriteUserItemDetailProviderRegistry the registry to register in
     * @param itemContext the item context to register for
     */
    protected AbstractFavouriteUserItemDetailProvider(
        @NonNull FavouriteUserItemDetailProviderRegistry favouriteUserItemDetailProviderRegistry,
        @NonNull FavouriteUserBsProperties favouriteUserBsProperties,
        @NonNull ItemContext itemContext) {
        Objects.requireNonNull(favouriteUserItemDetailProviderRegistry, "favouriteUserItemDetailProviderRegistry cannot be null");
        Objects.requireNonNull(favouriteUserBsProperties, "favouriteUserBsProperties cannot be null");
        Objects.requireNonNull(itemContext, "itemContext cannot be null");

        favouriteUserItemDetailProviderRegistry.register(itemContext, this);
        String contextKey = favouriteUserBsProperties.mapToContextKey(itemContext).orElse(null);
        contextProperties = Optional.ofNullable(favouriteUserBsProperties.context())
                                    .map(m -> m.get(contextKey))
                                    .orElse(FavouriteUserBsProperties.ContextProperties
                                        .builder()
                                        .filterEnabled(false)
                                        .build());

        filterProperties = favouriteUserBsProperties.filter();
    }

    /**
     * Returns the supported filter keys by the details provider.
     * By default, it does not support any filter keys.
     * The detail provider implementation should override and return the filter keys.
     *
     * @return Set of support filter keys
     */
    @NonNull
    protected Set<String> getSupportedFilterKeys() {
        return Set.of();
    }

    /**
     * Returns the filter properties related to context, if available.
     *
     * @return the available Filter Properties, defaults otherwise
     */
    @NonNull
    protected final FilterProperties getFilterProperties() {
        return filterProperties;
    }

    @NonNull
    @Override
    public final List<DetailRequest.Filter> getApplicableFilters(String filters) {
        FilterProperties filterProperties = getFilterProperties();
        final Set<String> supportedFilterKeys = getSupportedFilterKeys();

        // if filtering is not enabled (by default disabled),
        // or there are no supported filter keys
        // then return empty list
        if (!contextProperties.filterEnabled() || supportedFilterKeys.isEmpty()) {
            return List.of();
        }

        String valueSeparator = filterProperties.valueSeparator();
        return FilterUtils.buildQueryFilters(
                              filters,
                              filterProperties.separator(),
                              filterProperties.keyValueSeparator(),
                              valueSeparator,
                              null
                          )
                          .stream()
                          .filter(qf -> {
                              if (supportedFilterKeys.contains(qf.fieldName())) {
                                  return true;
                              } else {
                                  log.warn("Field {} is ignored as it is not supported for filtering", qf.fieldName());
                                  return false;
                              }
                          })
                          .map(qf -> DetailRequest.Filter
                              .builder()
                              .key(qf.fieldName())
                              .value(FilterUtils.combineValues(valueSeparator, qf.filterValues()))
                              .build())
                          .toList();
    }
}
