package com.finconsgroup.itserr.marketplace.institutional_page.bs.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.enums.SortDirection;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2AuthenticationException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2BusinessException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2FeignClientException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.InstitutionalPageDmClient;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.CreateInstitutionalPageIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.InstitutionalPageIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.InstitutionalPageViewIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.ModerationIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.OutputRequestUpdateDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.UpdateInstitutionalPageIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputCreateInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputModerationStatusDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputSearchForMemberInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputUpdateInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InstitutionalPageView;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputRequestUpdateDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputWorkspaceItemDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.enums.OperationType;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.exceptions.WP2InvalidImageException;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.mapper.InstitutionalPageMapper;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.InstitutionalPageProducer;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.NotificationMessageFactory;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.NotificationProducer;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.InstitutionalPageCreatedNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.InstitutionalPageDeletedNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.InstitutionalPageModerationRequestNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.InstitutionalPagePublicationRequestNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.InstitutionalPageStatusChangeNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.InstitutionalPageUpdatedNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.repository.UserProfileRepository;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.repository.WorkspaceRepository;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.service.InstitutionalPageService;
import com.finconsgroup.itserr.messaging.dto.MessagingEventDto;
import feign.FeignException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.net.URI;
import java.util.List;
import java.util.Set;
import java.util.UUID;

@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultInstitutionalPageService implements InstitutionalPageService {
    private final InstitutionalPageMapper institutionalPageMapper;
    private final InstitutionalPageDmClient institutionalPageDmClient;

    private final WorkspaceRepository workspaceRepository;
    private final UserProfileRepository userProfileRepository;
    
    private final NotificationMessageFactory notificationMessageFactory;
    private final NotificationProducer notificationProducer;

    private final InstitutionalPageProducer institutionalPageProducer;

    @Override
    public OutputPageDto<OutputInstitutionalPageDto> getAllInstitutionalPages(@NonNull InstitutionalPageView view, boolean includePublishedAndNotMember, Set<String> associationsToLoad, int pageNumber, int pageSize, String sort, SortDirection direction) {
        try {
            InstitutionalPageViewIPDmDto viewIPDmDto = institutionalPageMapper.toInstitutionalPageViewIPDmDto(view);
            OutputPageDto<InstitutionalPageIPDmDto> institutionalPagesDmDto = institutionalPageDmClient.getAllInstitutionalPages(
                    viewIPDmDto, includePublishedAndNotMember, associationsToLoad, pageNumber, pageSize, sort, direction);

            return institutionalPageMapper.toOutputPageDto(institutionalPagesDmDto);
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e);
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    @Override
    public OutputInstitutionalPageDto getInstitutionalPageById(UUID institutionalPageId, @NonNull InstitutionalPageView view, boolean includePublishedAndNotMember) {
        try {
            InstitutionalPageViewIPDmDto viewIPDmDto = institutionalPageMapper.toInstitutionalPageViewIPDmDto(view);
            InstitutionalPageIPDmDto institutionalPageDmDto = institutionalPageDmClient.getInstitutionalPageById(institutionalPageId, viewIPDmDto, includePublishedAndNotMember);

            return institutionalPageMapper.toOutputInstitutionalPageDto(institutionalPageDmDto, true);
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e);
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(e);
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    @Override
    public OutputInstitutionalPageDto createInstitutionalPage(InputCreateInstitutionalPageDto institutionalPageDto, MultipartFile imageFile) {
        try {
            // Validate image upload requirements
            if (imageFile != null && !imageFile.isEmpty()) {
                if (institutionalPageDto.getImageMetadata() == null || 
                    institutionalPageDto.getImageMetadata().getName() == null || 
                    institutionalPageDto.getImageMetadata().getName().trim().isEmpty()) {
                    throw new WP2InvalidImageException("Image name is required when uploading an image file");
                }
            }

            // Create the institutional page folder
            UUID institutionalPageFolderId = workspaceRepository.createInstitutionalPageFolder(institutionalPageDto.getName());

            // Handle image upload if provided
            String imageUrl = null;
            if (imageFile != null && !imageFile.isEmpty()) {
                try {
                    String imageDescription = institutionalPageDto.getImageMetadata().getDescription() != null 
                        ? institutionalPageDto.getImageMetadata().getDescription() 
                        : "";
                    String internalResourcesPrefix = getInternalResourcesPrefix(institutionalPageFolderId);
                    String imageName = internalResourcesPrefix + institutionalPageDto.getImageMetadata().getName();
                    URI uploadedImageUri = workspaceRepository.uploadInstitutionalPageImage(
                            institutionalPageFolderId,
                            imageFile.getInputStream(),
                            imageName,
                            imageDescription
                    );
                    imageUrl = uploadedImageUri.toString();
                } catch (IOException e) {
                    log.error("Failed to upload image for institutional page", e);
                    throw new WP2BusinessException("Failed to upload image: " + e.getMessage());
                }
            }

            // Creating the institutional page by calling the dm service
            CreateInstitutionalPageIPDmDto createInstitutionalPageDmDto = institutionalPageMapper.toCreateInstitutionalPageDmDto(
                    institutionalPageDto,
                    institutionalPageFolderId,
                    imageUrl
            );
            InstitutionalPageIPDmDto institutionalPageDmDto = institutionalPageDmClient.createInstitutionalPage(createInstitutionalPageDmDto);

            OutputInstitutionalPageDto outputInstitutionalPageDto = institutionalPageMapper.toOutputInstitutionalPageDto(institutionalPageDmDto, false);

            // Send notification (create and produce)
            MessagingEventDto<InstitutionalPageCreatedNotificationData> notificationMessage = notificationMessageFactory.createInstitutionalPageCreatedNotification(outputInstitutionalPageDto);
            notificationProducer.publishInstitutionalPageCreatedNotification(notificationMessage);

            return outputInstitutionalPageDto;
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e);
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(e.getMessage());
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    @Override
    public OutputInstitutionalPageDto updateInstitutionalPage(UUID institutionalPageId, InputUpdateInstitutionalPageDto institutionalPageDto, MultipartFile imageFile) {
        try {
            // Validate image upload requirements
            if (imageFile != null && !imageFile.isEmpty()) {
                if (institutionalPageDto.getImageMetadata() == null || 
                    institutionalPageDto.getImageMetadata().getName() == null || 
                    institutionalPageDto.getImageMetadata().getName().trim().isEmpty()) {
                    throw new WP2InvalidImageException("Image name is required when uploading an image file");
                }
            }

            // Determine image URL to use
            String finalImageUrl = null;
            if (imageFile != null && !imageFile.isEmpty()) {
                // New image file is being uploaded
                try {
                    UUID institutionalPageFolderId = retrieveInstitutionalPageFolderId(institutionalPageId);

                    String imageDescription = institutionalPageDto.getImageMetadata().getDescription() != null
                            ? institutionalPageDto.getImageMetadata().getDescription()
                            : "";

                    URI uploadedImageUri = workspaceRepository.uploadInstitutionalPageImage(
                            institutionalPageFolderId,
                            imageFile.getInputStream(),
                            institutionalPageDto.getImageMetadata().getName(),
                            imageDescription
                    );
                    finalImageUrl = uploadedImageUri.toString();
                } catch (IOException e) {
                    log.error("Failed to upload image for institutional page update", e);
                    throw new WP2BusinessException("Failed to upload image: " + e.getMessage());
                }
            } else if (institutionalPageDto.getImageUrl() != null) {
                // Existing image URL is being kept/updated
                finalImageUrl = institutionalPageDto.getImageUrl();
            }

            // Update the institutional page by calling the dm service
            UpdateInstitutionalPageIPDmDto updateInstitutionalPageDmDto =
                    institutionalPageMapper.toUpdateInstitutionalPageDmDto(institutionalPageDto, finalImageUrl);
            // Verify that the maintainer is an existing user
            if (updateInstitutionalPageDmDto.getMaintainer() != null)
                userProfileRepository.getMemberProfilesOrThrow(List.of(updateInstitutionalPageDmDto.getMaintainer())).getFirst();
            // Update on DM service
            InstitutionalPageIPDmDto institutionalPageDmDto =
                    institutionalPageDmClient.updateInstitutionalPage(institutionalPageId, updateInstitutionalPageDmDto);

            OutputInstitutionalPageDto outputInstitutionalPageDto = institutionalPageMapper.toOutputInstitutionalPageDto(institutionalPageDmDto, true);

            // Send notification (create and produce)
            MessagingEventDto<InstitutionalPageUpdatedNotificationData> notificationMessage = notificationMessageFactory.createInstitutionalPageUpdatedNotification(outputInstitutionalPageDto);
            notificationProducer.publishInstitutionalPageUpdatedNotification(notificationMessage);

            // produce the message with the entity, related to the approved version, not the updated one.
            InstitutionalPageIPDmDto originalInstitutionalPageDmDto = null;
            try {
                originalInstitutionalPageDmDto = retrieveApprovedInstitutionalPageById(institutionalPageId);
            } catch (WP2FeignClientException exception) {
                if (!exception.getHttpStatus().equals(HttpStatus.NOT_FOUND)) {
                    throw exception;
                }
                // otherwise no action needed,
                // this exception means the institutional page is never approved before,
                // so we don't need to send any event to open search
            }
            if (originalInstitutionalPageDmDto != null) {
                OutputInstitutionalPageDto originalOutputInstitutionalPageDto = institutionalPageMapper.toOutputInstitutionalPageDto(originalInstitutionalPageDmDto, true);
                institutionalPageProducer.publishUpdateEvent(originalOutputInstitutionalPageDto);
            }

            return outputInstitutionalPageDto;
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e);
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(e.getMessage());
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    @NonNull
    @Override
    public OutputPageDto<OutputInstitutionalPageDto> search(@NonNull InstitutionalPageView view, @NonNull InputSearchForMemberInstitutionalPageDto inputSearchForMemberInstitutionalPageDto,
                                                            Set<String> associationsToLoad, int pageNumber, int pageSize,
                                                            String sort, SortDirection direction) {
        InstitutionalPageViewIPDmDto viewIPDmDto = institutionalPageMapper.toInstitutionalPageViewIPDmDto(view);
        OutputPageDto<InstitutionalPageIPDmDto> institutionalPagesDmDto = institutionalPageDmClient.search(
                viewIPDmDto, inputSearchForMemberInstitutionalPageDto, associationsToLoad, pageNumber, pageSize, sort, direction
        );

        return institutionalPageMapper.toOutputPageDto(institutionalPagesDmDto);
    }

    @Override
    public void deleteInstitutionalPage(UUID institutionalPageId) {
        try {
            InstitutionalPageIPDmDto institutionalPageIPDmDto =
                    institutionalPageDmClient.deleteInstitutionalPage(institutionalPageId);

            if (institutionalPageDmClient.isAdmin()) {
                // The deletion does not require approval, so directly send the status change notification
                InputModerationStatusDto inputModerationStatusDto = InputModerationStatusDto.builder()
                        .approved(true)
                        .message("")
                        .build();

                ModerationIPDmDto moderationIPDmDto = ModerationIPDmDto.builder()
                        .institutionalPage(institutionalPageIPDmDto)
                        .operationType(OperationType.DELETE)
                        .build();

                MessagingEventDto<InstitutionalPageStatusChangeNotificationData> notificationMessage = notificationMessageFactory.createInstitutionalPageStatusChangeNotification(
                        inputModerationStatusDto,
                        moderationIPDmDto
                );
                notificationProducer.publishInstitutionalPageStatusChangeNotification(notificationMessage);

                // Send the deletion event
                OutputInstitutionalPageDto institutionalPageDto = institutionalPageMapper.toOutputInstitutionalPageDto(institutionalPageIPDmDto, true);
                institutionalPageProducer.publishDeleteEvent(institutionalPageDto);

            } else {

                // produce the message with the entity, related to the approved version, not the updated one.
                InstitutionalPageIPDmDto originalInstitutionalPageDmDto = retrieveApprovedInstitutionalPageById(institutionalPageId);
                OutputInstitutionalPageDto originalOutputInstitutionalPageDto = institutionalPageMapper.toOutputInstitutionalPageDto(originalInstitutionalPageDmDto, true);
                institutionalPageProducer.publishUpdateEvent(originalOutputInstitutionalPageDto);

                // Send notification to admin (create and produce)
                MessagingEventDto<InstitutionalPageDeletedNotificationData> notificationMessage = notificationMessageFactory.createInstitutionalPageDeletedNotification(institutionalPageIPDmDto);
                notificationProducer.publishInstitutionalPageDeletedNotification(notificationMessage);
            }

        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e);
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(e.getMessage());
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    @NonNull
    @Override
    public OutputPageDto<OutputInstitutionalPageDto> findInstitutionalPagesHierarchyByRootId(
            @NonNull UUID rootInstitutionalPageId,
            Set<String> associationsToLoad,
            int pageNumber,
            int pageSize,
            String sort,
            SortDirection direction
    ) {
        try {
            OutputPageDto<InstitutionalPageIPDmDto> institutionalPagesDmDto = 
                    institutionalPageDmClient.findInstitutionalPagesHierarchyByRootId(
                            rootInstitutionalPageId, associationsToLoad, pageNumber, pageSize, sort, direction);

            return institutionalPagesDmDto.map(
                    institutionalPageDmDto -> institutionalPageMapper.toOutputInstitutionalPageDto(institutionalPageDmDto, true)
            );
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e);
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(rootInstitutionalPageId);
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    @Override
    public List<OutputWorkspaceItemDto> retrieveFolderContent(UUID institutionalPageId, int pageNumber, int pageSize) {
        try {
            UUID folderId = retrieveInstitutionalPageFolderId(institutionalPageId);
            String internalResourcesPrefix = getInternalResourcesPrefix(folderId);
            return workspaceRepository.getWorkspaceChildrenByFolderId(folderId, pageNumber, pageSize)
                    .stream()
                    .filter(workspaceBsDto ->
                            !workspaceBsDto.getName().startsWith(internalResourcesPrefix))
                    .map(institutionalPageMapper::toOutputWorkspaceItemDto)
                    .toList();
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e);
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(e.getMessage());
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    @Override
    public OutputInstitutionalPageDto requestModerationInstitutionalPage(UUID institutionalPageId) {
        try {
            InstitutionalPageIPDmDto institutionalPageDmDto =
                    institutionalPageDmClient.requestModerationInstitutionalPageById(institutionalPageId);

            OutputInstitutionalPageDto outputInstitutionalPageDto =
                    institutionalPageMapper.toOutputInstitutionalPageDto(institutionalPageDmDto, true);

            MessagingEventDto<InstitutionalPageModerationRequestNotificationData> notification =
                    notificationMessageFactory.createInstitutionalPageModerationRequestNotification(outputInstitutionalPageDto);
            notificationProducer.publishInstitutionalPageModerationRequestNotification(notification);

            // produce the message with the entity, related to the approved version, not the updated one.
            InstitutionalPageIPDmDto originalInstitutionalPageDmDto = null;
            try {
                originalInstitutionalPageDmDto = retrieveApprovedInstitutionalPageById(institutionalPageId);
            } catch (WP2FeignClientException exception) {
                if (!exception.getHttpStatus().equals(HttpStatus.NOT_FOUND)) {
                    throw exception;
                }
                // otherwise no action needed,
                // this exception means the institutional page is never approved before,
                // so we don't need to send any event to open search
            }
            if (originalInstitutionalPageDmDto != null) {
                OutputInstitutionalPageDto originalOutputInstitutionalPageDto = institutionalPageMapper.toOutputInstitutionalPageDto(originalInstitutionalPageDmDto, true);
                institutionalPageProducer.publishUpdateEvent(originalOutputInstitutionalPageDto);
            }

            return outputInstitutionalPageDto;
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e.getMessage());
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(e);
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    @Override
    public OutputInstitutionalPageDto requestPublicationInstitutionalPage(UUID institutionalPageId) {
        try {
            InstitutionalPageIPDmDto institutionalPageDmDto =
                    institutionalPageDmClient.requestPublicationInstitutionalPageById(institutionalPageId);

            OutputInstitutionalPageDto outputInstitutionalPageDto =
                    institutionalPageMapper.toOutputInstitutionalPageDto(institutionalPageDmDto, true);

            MessagingEventDto<InstitutionalPagePublicationRequestNotificationData> notification =
                    notificationMessageFactory.createInstitutionalPagePublicationRequestNotification(outputInstitutionalPageDto);
            notificationProducer.publishInstitutionalPagePublicationRequestNotification(notification);

            // produce the message with the entity, related to the approved version, not the updated one.
            InstitutionalPageIPDmDto originalInstitutionalPageDmDto = retrieveApprovedInstitutionalPageById(institutionalPageId);
            OutputInstitutionalPageDto originalOutputInstitutionalPageDto = institutionalPageMapper.toOutputInstitutionalPageDto(originalInstitutionalPageDmDto, true);
            institutionalPageProducer.publishUpdateEvent(originalOutputInstitutionalPageDto);

            return outputInstitutionalPageDto;
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e.getMessage());
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(e);
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }


    @NonNull
    @Override
    public OutputRequestUpdateDto requestUpdateInstitutionalPageById(@NonNull UUID institutionalPageId) {
        try {
            OutputRequestUpdateDmDto outputRequestUpdateDmDto = institutionalPageDmClient.requestUpdateInstitutionalPageById(institutionalPageId);
            OutputRequestUpdateDto outputRequestUpdateDto = institutionalPageMapper.toOutputRequestUpdateDto(outputRequestUpdateDmDto);
            return outputRequestUpdateDto;
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e);
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(e.getMessage());
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    @Override
    public void cancelUpdateInstitutionalPageById(@NonNull UUID institutionalPageId) {
        try {
            institutionalPageDmClient.cancelUpdateInstitutionalPageById(institutionalPageId);
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e);
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(e.getMessage());
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    private static String getInternalResourcesPrefix(UUID institutionalPageFolderId) {
        return Integer.toHexString(institutionalPageFolderId.hashCode()) + "_";
    }

    private UUID retrieveInstitutionalPageFolderId(UUID institutionalPageId) {
        return retrieveOriginalInstitutionalPageById(institutionalPageId).getWorkspaceFolderId();
    }

    private InstitutionalPageIPDmDto retrieveApprovedInstitutionalPageById(UUID institutionalPageId) {
        return institutionalPageDmClient.getInstitutionalPageById(institutionalPageId, InstitutionalPageViewIPDmDto.APPROVED, false);
    }

    private InstitutionalPageIPDmDto retrieveOriginalInstitutionalPageById(UUID institutionalPageId) {
        return institutionalPageDmClient.getInstitutionalPageById(institutionalPageId, InstitutionalPageViewIPDmDto.ORIGINAL, false);
    }
}
