package com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.enums.SortDirection;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.CreateInstitutionalPageIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.CreateParagraphIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.InstitutionalPageIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.InstitutionalPageViewIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.InviteMembersIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.MembersInHierarchyIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.ModerationIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.ModerationStatusDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.OutputRequestUpdateDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.PatchIPInvitationRequestIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.PatchIPJoinRequestIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.PatchMembershipIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.PendingMemberRequestsIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.RemoveMembershipIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.SubmitJoinRequestIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.UpdateInstitutionalPageIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.UpdateParagraphIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputSearchForMemberInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputSearchModerationInstitutionalPageDto;
import jakarta.validation.Valid;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.Set;
import java.util.UUID;

public interface InstitutionalPageDmClient {

    /**
     * Get institutional page by id
     * @param institutionalPageId the id of the institutional page
     * @param view the selection strategy to apply
     * @return the institutional page dto
     */
    @GetMapping("/institutional-page/institutional-pages/{institutionalPageId}")
    InstitutionalPageIPDmDto getInstitutionalPageById(
            @PathVariable UUID institutionalPageId,
            @RequestParam(name = "view", required = false) InstitutionalPageViewIPDmDto view,
            @RequestParam(name = "includePublishedAndNotMember", required = false) Boolean includePublishedAndNotMember
    );

    /**
     * Get all institutional pages for which the user is a contributor
     * @param view the selection strategy to apply
     * @param pageNumber the page number
     * @param pageSize the size of the page
     * @param sort the attribute to sort by
     * @param direction the sorting direction (ASC or DESC)
     * @return the dto of the institutional page
     */
    @GetMapping("/institutional-page/institutional-pages")
    OutputPageDto<InstitutionalPageIPDmDto> getAllInstitutionalPages(
            @RequestParam(name = "view", required = false) InstitutionalPageViewIPDmDto view,
            @RequestParam(name = "includePublishedAndNotMember", required = false) Boolean includePublishedAndNotMember,
            @RequestParam(name = "associationsToLoad") Set<String> associationsToLoad,
            @RequestParam(name = "pageNumber") int pageNumber,
            @RequestParam(name = "pageSize") int pageSize,
            @RequestParam(name ="sort") String sort,
            @RequestParam(name = "direction") SortDirection direction
    );

    /**
     * Creates an institutional page
     *
     * @param institutionalPageDmDto the institutional page to create
     * @return the institutional page created
     * @throws feign.FeignException.Unauthorized if the user is not authenticated
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     * @throws feign.FeignException.NotFound if some resource in the dto is not found
     */
    @PostMapping("/institutional-page/institutional-pages")
    InstitutionalPageIPDmDto createInstitutionalPage(@RequestBody CreateInstitutionalPageIPDmDto institutionalPageDmDto);

    /**
     * Updates an institutional page
     *
     * @param institutionalPageId the id of the institutional page to update
     * @param institutionalPageDmDto the institutional page to update
     * @return the institutional page updated
     * @throws feign.FeignException.Unauthorized if the user is not authenticated
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     * @throws feign.FeignException.NotFound if the institutional page does not exist
     */
    @PutMapping("/institutional-page/institutional-pages/{institutionalPageId}")
    InstitutionalPageIPDmDto updateInstitutionalPage(
            @PathVariable UUID institutionalPageId,
            @RequestBody UpdateInstitutionalPageIPDmDto institutionalPageDmDto);

    /**
     * Deletes an institutional page
     *
     * @param institutionalPageId the id of the institutional page to delete
     * @throws feign.FeignException.Unauthorized if the user is not authenticated
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     * @throws feign.FeignException.NotFound if the institutional page does not exist
     *
     * @return the institutional page to be deleted
     */
    @DeleteMapping("/institutional-page/institutional-pages/{institutionalPageId}")
    InstitutionalPageIPDmDto deleteInstitutionalPage(@PathVariable UUID institutionalPageId);

    /**
     * Retrieves a paginated list of all InstitutionalPages matching the search criteria.
     *
     * @param view the selection strategy to apply
     * @param inputSearchForMemberInstitutionalPageDto the dto containing the filters to be applied
     * @param pageNumber                      the page number to retrieve (default is 0)
     * @param pageSize                        the number of InstitutionalPages per page (default is 10)
     * @param sort                            the field to sort by (default is "id")
     * @param direction                       the direction of sorting (default is ascending)
     * @return a page of {@link InstitutionalPageIPDmDto} and HTTP status 200 (OK)
     */
    @PostMapping(value = "/institutional-page/institutional-pages/search", consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE)
    OutputPageDto<InstitutionalPageIPDmDto> search(
            @RequestParam(name = "view", required = false) InstitutionalPageViewIPDmDto view,
            @RequestBody InputSearchForMemberInstitutionalPageDto inputSearchForMemberInstitutionalPageDto,
            @RequestParam(name = "associationsToLoad") Set<String> associationsToLoad,
            @RequestParam(name = "pageNumber") int pageNumber,
            @RequestParam(name = "pageSize") int pageSize,
            @RequestParam(name = "sort") String sort,
            @RequestParam(name = "direction") SortDirection direction
    );

    /**
     * Create a paragraph for an institutional page
     *
     * @param institutionalPageId the id of the institutional page
     * @param paragraphDto the paragraph to create
     * @return the institutional page with the created paragraph
     * @throws feign.FeignException.Unauthorized if the user is not authenticated
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     * @throws feign.FeignException.NotFound if the institutional page does not exist
     */
    @PostMapping("/institutional-page/institutional-pages/{institutionalPageId}/paragraphs")
    InstitutionalPageIPDmDto createParagraph(
            @PathVariable UUID institutionalPageId,
            @RequestBody CreateParagraphIPDmDto paragraphDto,
            @RequestParam Integer position
    );

    /**
     * Update a paragraph of an institutional page
     *
     * @param institutionalPageId the id of the institutional page
     * @param paragraphId the id of the paragraph to update
     * @param paragraphDto the paragraph to update
     * @return the institutional page with the updated paragraph
     * @throws feign.FeignException.Unauthorized if the user is not authenticated
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     * @throws feign.FeignException.NotFound if the institutional page or the paragraph does not exist
     */
    @PutMapping("/institutional-page/institutional-pages/{institutionalPageId}/paragraphs/{paragraphId}")
    InstitutionalPageIPDmDto updateParagraph(
            @PathVariable UUID institutionalPageId,
            @PathVariable UUID paragraphId,
            @RequestBody UpdateParagraphIPDmDto paragraphDto
    );

    /**
     * Delete a paragraph of an institutional page
     *
     * @param institutionalPageId the id of the institutional page
     * @param paragraphId the id of the paragraph to delete
     * @return the institutional page with the deleted paragraph
     * @throws feign.FeignException.Unauthorized if the user is not authenticated
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     * @throws feign.FeignException.NotFound if the institutional page or the paragraph does not exist
     */
    @DeleteMapping("/institutional-page/institutional-pages/{institutionalPageId}/paragraphs/{paragraphId}")
    InstitutionalPageIPDmDto deleteParagraph(
            @PathVariable UUID institutionalPageId,
            @PathVariable UUID paragraphId
    );

    /**
     * Request moderation of an institutional page
     *
     * @param institutionalPageId the id of the institutional page
     * @return the institutional page for which the request is made
     * @throws feign.FeignException.Unauthorized if the user is not authorized to perform this operation
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     * @throws feign.FeignException.NotFound if the institutional page does not exist
     */
    @PutMapping("/institutional-page/institutional-pages/{institutionalPageId}/request-moderation")
    InstitutionalPageIPDmDto requestModerationInstitutionalPageById(@PathVariable UUID institutionalPageId);

    /**
     * Request publication of an institutional page
     *
     * @param institutionalPageId the id of the institutional page
     * @return the institutional page for which the request is made
     * @throws feign.FeignException.Unauthorized if the user is not authorized to perform this operation
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     * @throws feign.FeignException.NotFound if the institutional page does not exist
     */
    @PutMapping("/institutional-page/institutional-pages/{institutionalPageId}/request-publication")
    InstitutionalPageIPDmDto requestPublicationInstitutionalPageById(@PathVariable UUID institutionalPageId);

    /**
     * Retrieves a paginated list of all InstitutionalPages under moderation (Pending or Rejected) - moderator only
     *
     * @param pageNumber the page number
     * @param pageSize the size of the page
     * @param sort the attribute to sort by
     * @param direction the sorting direction (ASC or DESC)
     * @return the institutional pages under moderation
     * @throws feign.FeignException.Unauthorized if the user is not authorized to perform this operation
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     */
    @GetMapping(path = "/institutional-page/institutional-pages/moderation", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputPageDto<InstitutionalPageIPDmDto> getModerationInstitutionalPages(
            @RequestParam(name = "associationsToLoad") Set<String> associationsToLoad,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) int pageSize,
            @RequestParam(name ="sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) SortDirection direction
    );

    /**
     * Retrieves a InstitutionalPage under moderation (Pending or Rejected) by id - moderator only
     *
     * @param institutionalPageId the id of the institutional page
     * @return the institutional page under moderation
     * @throws feign.FeignException.Unauthorized if the user is not authorized to perform this operation
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     * @throws feign.FeignException.NotFound if the institutional page does not exist
     */
    @GetMapping("/institutional-page/institutional-pages/moderation/{institutionalPageId}")
    InstitutionalPageIPDmDto getModerationInstitutionalPageById(@PathVariable UUID institutionalPageId);

    /**
     * Approve or reject an institutional page
     *
     * @param institutionalPageId the id of the institutional page
     * @return the approved/rejected institutional page
     * @throws feign.FeignException.Unauthorized if the user is not authorized to perform this operation
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     * @throws feign.FeignException.NotFound if the institutional page does not exist
     */
    @PatchMapping("/institutional-page/institutional-pages/moderation/{institutionalPageId}")
    ModerationIPDmDto changeInstitutionalPageModerationStatus(
            @PathVariable UUID institutionalPageId,
            @Valid @RequestBody ModerationStatusDmDto inputModerationStatusDto);

    /**
     * Retrieves all pending member join requests and invitations for a specific InstitutionalPage (WP Leader).
     *
     * @param institutionalPageId the id of the InstitutionalPage
     * @return {@link PendingMemberRequestsIPDmDto} and HTTP status 200 (OK)
     * @throws feign.FeignException.Unauthorized if the user is not authorized to perform this operation
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     * @throws feign.FeignException.NotFound if the institutional page does not exist
     */
    @GetMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/members/requests",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    PendingMemberRequestsIPDmDto findPendingMemberRequestsForInstitutionalPage(
            @PathVariable("institutionalPageId") UUID institutionalPageId
    );

    /**
     * Retrieves all pending member join requests and invitations related to the authenticated user.
     *
     * @return {@link PendingMemberRequestsIPDmDto} and HTTP status 200 (OK)
     * @throws feign.FeignException.Unauthorized if the user is not authorized to perform this operation
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     */
    @GetMapping(
            value = "/institutional-page/institutional-pages/members/requests",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    PendingMemberRequestsIPDmDto findPendingMemberRequestsForUser();

    /**
     * Invites users to join the specified InstitutionalPage as members (WP Leader).
     *
     * @param institutionalPageId   the id of the InstitutionalPage
     * @param inviteMembersIPDmDto the DTO for inviting users to join the InstitutionalPage as members
     * @return {@link PendingMemberRequestsIPDmDto} and HTTP status 200 (OK)
     * @throws feign.FeignException.Unauthorized if the user is not authorized to perform this operation
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     * @throws feign.FeignException.NotFound if the institutional page does not exist
     */
    @PostMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/members/requests/invitations",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    PendingMemberRequestsIPDmDto inviteMembers(
            @PathVariable("institutionalPageId") UUID institutionalPageId,
            @Valid @RequestBody InviteMembersIPDmDto inviteMembersIPDmDto
    );

    /**
     * Cancels an invitation for a specific user to join an InstitutionalPage (WP Leader).
     *
     * @param institutionalPageId the id of the InstitutionalPage
     * @param invitedUserId       the id of the invited user
     * @return {@link PendingMemberRequestsIPDmDto} and HTTP status 200 (OK)
     * @throws feign.FeignException.Unauthorized if the user is not authorized to perform this operation
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     * @throws feign.FeignException.NotFound if the institutional page does not exist
     */
    @DeleteMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/members/requests/invitations/{invitedUserId}",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    PendingMemberRequestsIPDmDto cancelInvitation(
            @PathVariable("institutionalPageId") UUID institutionalPageId,
            @PathVariable("invitedUserId") UUID invitedUserId
    );

    /**
     * Accepts/Rejects an invitation to join an InstitutionalPage
     *
     * @param institutionalPageId              the id of the InstitutionalPage
     * @param patchIPInvitationRequestDto the DTO to approve or reject invitation to an Institutional Page
     * @return {@link PendingMemberRequestsIPDmDto} and HTTP status 200 (OK)
     * @throws feign.FeignException.Unauthorized if the user is not authorized to perform this operation
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     * @throws feign.FeignException.NotFound if the institutional page does not exist
     */
    @PatchMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/members/requests/invitations",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    PendingMemberRequestsIPDmDto acceptOrRejectInvitation(
            @PathVariable("institutionalPageId") UUID institutionalPageId,
            @Valid @RequestBody PatchIPInvitationRequestIPDmDto patchIPInvitationRequestDto
    );

    /**
     * Updates membership of members to WP Leader/Member for an InstitutionalPage (WP Leader).
     *
     * @param institutionalPageId     the id of the InstitutionalPage
     * @param patchMembershipDto Input DTO for updating membership of users
     * @return {@link InstitutionalPageIPDmDto} and HTTP status 200 (OK)
     * @throws feign.FeignException.Unauthorized if the user is not authorized to perform this operation
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     * @throws feign.FeignException.NotFound if the institutional page does not exist
     */
    @PatchMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/members/membership",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    InstitutionalPageIPDmDto patchMembershipOfUsersForInstitutionalPage(
            @PathVariable("institutionalPageId") UUID institutionalPageId,
            @Valid @RequestBody PatchMembershipIPDmDto patchMembershipDto
    );

    /**
     * Remove membership of users for an InstitutionalPage.
     *
     * @param institutionalPageId      the id of the InstitutionalPage
     * @param removeMembershipDto Input DTO for removing membership of users
     * @return {@link InstitutionalPageIPDmDto} and HTTP status 200 (OK)
     * @throws feign.FeignException.Unauthorized if the user is not authorized to perform this operation
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     * @throws feign.FeignException.NotFound if the institutional page does not exist
     */
    @DeleteMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/members/membership",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    InstitutionalPageIPDmDto removeMembershipOfUsersForInstitutionalPage(
            @PathVariable("institutionalPageId") UUID institutionalPageId,
            @Valid @RequestBody RemoveMembershipIPDmDto removeMembershipDto
    );

    /**
     * Submits a join request for the authenticated user to join an InstitutionalPage.
     *
     * @param institutionalPageId       the id of the InstitutionalPage
     * @param submitJoinRequestDto the DTO Input DTO for submitting a join request
     * @return {@link PendingMemberRequestsIPDmDto} and HTTP status 200 (OK)
     * @throws feign.FeignException.Unauthorized if the user is not authorized to perform this operation
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     * @throws feign.FeignException.NotFound if the institutional page does not exist
     */
    @PostMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/members/requests/join",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    PendingMemberRequestsIPDmDto submitJoinRequest(
            @PathVariable("institutionalPageId") UUID institutionalPageId,
            @Valid @RequestBody SubmitJoinRequestIPDmDto submitJoinRequestDto
    );

    /**
     * Cancels a join request previously submitted by the authenticated user.
     *
     * @param institutionalPageId the id of the InstitutionalPage
     * @return {@link PendingMemberRequestsIPDmDto} and HTTP status 200 (OK)
     * @throws feign.FeignException.Unauthorized if the user is not authorized to perform this operation
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     * @throws feign.FeignException.NotFound if the institutional page does not exist
     */
    @DeleteMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/members/requests/join",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    PendingMemberRequestsIPDmDto cancelJoinRequest(
            @PathVariable("institutionalPageId") UUID institutionalPageId
    );

    /**
     * Accepts/Rejects join requests to an institutional page (WP Leader).
     *
     * @param institutionalPageId        the id of the InstitutionalPage
     * @param patchIPJoinRequestDto the DTO to approve or reject a pending join request to an Institutional Page
     * @return {@link PendingMemberRequestsIPDmDto} and HTTP status 200 (OK)
     * @throws feign.FeignException.Unauthorized if the user is not authorized to perform this operation
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     * @throws feign.FeignException.NotFound if the institutional page does not exist
     */
    @PatchMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/members/requests/join",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    PendingMemberRequestsIPDmDto acceptOrRejectJoinRequests(
            @PathVariable("institutionalPageId") UUID institutionalPageId,
            @Valid @RequestBody PatchIPJoinRequestIPDmDto patchIPJoinRequestDto
    );


    /**
     * Retrieves a paginated list of all InstitutionalPages the user is contributing to,
     * corresponding to the hierarchy associated with the rootInstitutionalPageId.
     *
     * @param associationsToLoad comma separated list of the associations to be returned (default is "all").
     * @param pageNumber         the page number to retrieve (default is 0)
     * @param pageSize           the number of InstitutionalPages per page (default is 10)
     * @param sort               the field to sort by (default is "id")
     * @param direction          the direction of sorting (default is ascending)
     * @return a page of {@link InstitutionalPageIPDmDto} and HTTP status 200 (OK)
     */
    @GetMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/hierarchy",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    OutputPageDto<InstitutionalPageIPDmDto> findInstitutionalPagesHierarchyByRootId(
            @PathVariable("institutionalPageId") UUID rootInstitutionalPageId,
            @RequestParam(name = "associationsToLoad", defaultValue = "all", required = false) Set<String> associationsToLoad,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) int pageSize,
            @RequestParam(name = "sort", defaultValue = "id", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) SortDirection direction
    );

    /**
     * Retrieves paginated list of all users belonging
     * to the hierarchy of a given root InstitutionalPage (WP Leader).
     *
     * @param rootInstitutionalPageId ID of the root institutional page
     * @param pageNumber              the page number to retrieve (default is 0)
     * @param pageSize                the number of users per page (default is 10)
     * @return a page of {@link MembersInHierarchyIPDmDto} and HTTP status 200 (OK)
     * @throws feign.FeignException.Unauthorized if the user is not authorized to perform this operation
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     * @throws feign.FeignException.NotFound if the institutional page does not exist
     */
    @GetMapping(
            value = "/institutional-page/institutional-pages/{rootInstitutionalPageId}/members/hierarchy",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    OutputPageDto<MembersInHierarchyIPDmDto> findAllUsersInHierarchy(
            @PathVariable("rootInstitutionalPageId") UUID rootInstitutionalPageId,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) int pageSize
    );

    /**
     * Retrieves a paginated list of all InstitutionalPages under moderation (Pending or Rejected), matching the search criteria - moderator only
     *
     * @param inputSearchModerationInstitutionalPageDto the search criteria
     * @param associationsToLoad the associations to be returned on the response
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize the number of InstitutionalPages per page (default is 10)
     * @param sort the field to sort by (default is "name")
     * @param direction the direction of sorting (default is ascending)
     * @return a page of {@link InstitutionalPageIPDmDto} and HTTP status 200 (OK)
     * @throws feign.FeignException.Unauthorized if the user is not authorized to perform this operation
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     */
    @PostMapping(value = "/institutional-page/institutional-pages/moderation/search", consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE)
    OutputPageDto<InstitutionalPageIPDmDto> searchModerationInstitutionalPages(
            @RequestBody InputSearchModerationInstitutionalPageDto inputSearchModerationInstitutionalPageDto,
            @RequestParam(name = "associationsToLoad") Set<String> associationsToLoad,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) int pageSize,
            @RequestParam(name = "sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) SortDirection direction
    );

    /**
     * Indicates whether the user has Admin role.
     *
     * @return {@link Boolean} and HTTP status 200 (OK)
     */
    @GetMapping(
            value = "/institutional-page/admin",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    Boolean isAdmin();


    /**
     * Updates the folder ID associated with an institutional page.
     * Normally, the folder ID should never change, but this endpoint is required to handle
     * a specific d4science behavior where the workspace folder ID is updated after a folder is unshared.
     *
     * @param institutionalPageId the id of the InstitutionalPage to update
     * @param newFolderId         the new id of the folder
     * @return the updated {@link InstitutionalPageIPDmDto} and HTTP status 200 (OK)
     */
    @PutMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/folders/{newFolderId}",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    InstitutionalPageIPDmDto updateFolderId(
            @PathVariable("institutionalPageId") UUID institutionalPageId,
            @PathVariable("newFolderId") UUID newFolderId
    );

    /**
     * Request update for a specific InstitutionalPage by id, only if the user is a member
     *
     * @param institutionalPageId the id of the InstitutionalPage to update
     * @return the updated {@link OutputRequestUpdateDmDto} and HTTP status 200 (OK)
     */
    @PutMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/request-update",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    OutputRequestUpdateDmDto requestUpdateInstitutionalPageById(
            @PathVariable("institutionalPageId") UUID institutionalPageId
    );

    /**
     * Release lock-for-update of a specific InstitutionalPage by id, only if the user is a member.
     *
     * @param institutionalPageId the id of the InstitutionalPage to update
     */
    @PutMapping(value = "/institutional-page/institutional-pages/{institutionalPageId}/cancel-update")
    void cancelUpdateInstitutionalPageById(
            @PathVariable("institutionalPageId") UUID institutionalPageId
    );

}
