package com.finconsgroup.itserr.marketplace.institutionalpage.dm.api;

import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputCreateInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputSearchForMemberInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputUpdateInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputRequestUpdateDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.validation.ValidAssociationToLoad;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.Positive;
import jakarta.validation.constraints.PositiveOrZero;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Sort;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.Set;
import java.util.UUID;

/**
 * This interface defines the contract for REST API endpoints related to InstitutionalPages Management.
 *
 * <p>
 * It provides endpoints for creating, retrieving, searching, updating and deleting InstitutionalPages.
 * </p>
 *
 * <p>Example usage:
 * <pre>
 * POST   /api/v1/dm/institutional-page/institutional-pages                       - Creates a new InstitutionalPage.
 * GET    /api/v1/dm/institutional-page/institutional-pages                       - Retrieves a paginated list of all InstitutionalPages the user is contributing to.
 * POST   /api/v1/dm/institutional-page/institutional-pages/search                - Retrieves a paginated list of all InstitutionalPages the user is contributing to, matching the search criteria..
 * PUT    /api/v1/dm/institutional-page/institutional-pages/{institutionalPageId} - Updates a specific InstitutionalPage by id, only if the user is a member.
 * GET    /api/v1/dm/institutional-page/institutional-pages/{institutionalPageId} - Retrieve a specific InstitutionalPage by id, only if the user is a member.
 * DELETE /api/v1/dm/institutional-page/institutional-pages/{institutionalPageId} - Deletes a specific InstitutionalPage by id, only if the user is a member.
 * GET    /api/v1/dm/institutional-page/institutional-pages/{institutionalPageId}/hierarchy - Retrieves a paginated list of all InstitutionalPages the user is contributing to, corresponding to the hierarchy associated to the rootInstitutionalPageId
 * PUT    /api/v1/dm/institutional-page/institutional-pages/{institutionalPageId}/folders/{newFolderId} - Updates the folder ID associated with an institutional page
 * PUT    /api/v1/dm/institutional-page/institutional-pages/{institutionalPageId}/request-update - Request update for a specific InstitutionalPage by id, only if the user is a member.
 * PUT    /api/v1/dm/institutional-page/institutional-pages/{institutionalPageId}/cancel-update - Release lock-for-update of a specific InstitutionalPage by id, only if the user is a member.
 * </pre>
 * </p>
 */
@Tag(
        name = "InstitutionalPage",
        description = "The InstitutionalPage API: it provides endpoints for " +
                "creating, retrieving, searching, updating and deleting InstitutionalPages."
)
@SecurityRequirement(name = "BearerAuth")
public interface InstitutionalPageApi {

    /**
     * Creates a new InstitutionalPage.
     *
     * @param inputCreateInstitutionalPageDto the input data transfer object containing InstitutionalPage details
     * @return the created {@link OutputInstitutionalPageDto} and HTTP status 201 (Created)
     */
    @Operation(
            summary = "Creates a new InstitutionalPage",
            responses = {
                    @ApiResponse(responseCode = "201", description = "Created")
            }
    )
    @PostMapping(
            value = "/institutional-page/institutional-pages",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.CREATED)
    OutputInstitutionalPageDto createInstitutionalPage(
            @Valid @RequestBody InputCreateInstitutionalPageDto inputCreateInstitutionalPageDto
    );

    /**
     * Retrieves a paginated list of all InstitutionalPages the user is contributing to.
     *
     * @param associationsToLoad comma separated list of the associations to be returned (default is "all").
     * @param pageNumber         the page number to retrieve (default is 0)
     * @param pageSize           the number of InstitutionalPages per page (default is 10)
     * @param sort               the field to sort by (default is "name")
     * @param direction          the direction of sorting (default is ascending)
     * @return a page of {@link OutputInstitutionalPageDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Retrieves a paginated list of all InstitutionalPages the user is contributing to",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @GetMapping(value = "/institutional-page/institutional-pages", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    Page<OutputInstitutionalPageDto> findAllInstitutionalPages(
            @RequestParam(name = "associationsToLoad", defaultValue = "all", required = false) Set<@ValidAssociationToLoad String> associationsToLoad,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction
    );

    /**
     * Retrieves a paginated list of all InstitutionalPages the user is contributing to, matching the search criteria.
     *
     * @param inputSearchForMemberInstitutionalPageDto the dto containing the filters to be applied
     * @param associationsToLoad                       comma separated list of the associations to be returned (default is "all").
     * @param pageNumber                               the page number to retrieve (default is 0)
     * @param pageSize                                 the number of InstitutionalPages per page (default is 10)
     * @param sort                                     the field to sort by (default is "name")
     * @param direction                                the direction of sorting (default is ascending)
     * @return a page of {@link OutputInstitutionalPageDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Retrieves a paginated list of all InstitutionalPages the user is contributing to, matching the search criteria",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @PostMapping(
            value = "/institutional-page/institutional-pages/search",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    Page<OutputInstitutionalPageDto> search(
            @Valid @RequestBody InputSearchForMemberInstitutionalPageDto inputSearchForMemberInstitutionalPageDto,
            @RequestParam(name = "associationsToLoad", defaultValue = "all", required = false) Set<@ValidAssociationToLoad String> associationsToLoad,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction
    );

    /**
     * Retrieves a specific InstitutionalPage by id, only if the user is a member.
     *
     * @param institutionalPageId the id of the InstitutionalPage to retrieve
     * @return the found {@link OutputInstitutionalPageDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Retrieves a specific InstitutionalPage by id, only if the user is a member",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found"),
            }
    )
    @GetMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputInstitutionalPageDto findInstitutionalPageById(@PathVariable("institutionalPageId") UUID institutionalPageId);

    /**
     * Updates a specific InstitutionalPage by id, only if the user is a member.
     *
     * @param institutionalPageId             the id of the InstitutionalPage to update
     * @param inputUpdateInstitutionalPageDto the input data transfer object containing updated InstitutionalPage details
     * @return the updated {@link OutputInstitutionalPageDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Updates a specific InstitutionalPage by id, only if the user is a member",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found"),
            }
    )
    @PutMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputInstitutionalPageDto updateInstitutionalPageById(
            @PathVariable("institutionalPageId") UUID institutionalPageId,
            @Valid @RequestBody InputUpdateInstitutionalPageDto inputUpdateInstitutionalPageDto
    );

    /**
     * Deletes a specific InstitutionalPage by id, only if the user is a member.
     *
     * @param institutionalPageId the id of the InstitutionalPage to delete
     */
    @Operation(
            summary = "Deletes a specific InstitutionalPage by id, only if the user is a member",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found"),
            }
    )
    @DeleteMapping(value = "/institutional-page/institutional-pages/{institutionalPageId}")
    @ResponseStatus(HttpStatus.OK)
    OutputInstitutionalPageDto deleteInstitutionalPageById(
            @PathVariable("institutionalPageId") UUID institutionalPageId
    );

    /**
     * Retrieves a paginated list of all InstitutionalPages the user is contributing to,
     * corresponding to the hierarchy associated to the rootInstitutionalPageId.
     *
     * @param associationsToLoad comma separated list of the associations to be returned (default is "all").
     * @param pageNumber         the page number to retrieve (default is 0)
     * @param pageSize           the number of InstitutionalPages per page (default is 10)
     * @param sort               the field to sort by (default is "name")
     * @param direction          the direction of sorting (default is ascending)
     * @return a page of {@link OutputInstitutionalPageDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Retrieves a paginated list of all InstitutionalPages the user is contributing to, " +
                    "corresponding to the hierarchy associated to the rootInstitutionalPageId",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @GetMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/hierarchy",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    Page<OutputInstitutionalPageDto> findInstitutionalPagesHierarchyByRootId(
            @PathVariable("institutionalPageId") UUID rootInstitutionalPageId,
            @RequestParam(name = "associationsToLoad", defaultValue = "all", required = false) Set<@ValidAssociationToLoad String> associationsToLoad,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction
    );

    /**
     * Updates the folder ID associated with an institutional page, only if the user is a member.
     * Normally, the folder ID should never change, but this endpoint is required to handle
     * a specific d4science behavior where the workspace folder ID is updated after a folder is unshared.
     *
     * @param institutionalPageId the id of the InstitutionalPage to update
     * @param newFolderId         the new id of the folder
     * @return the updated {@link OutputInstitutionalPageDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Updates the folder ID associated with an institutional page, only if the user is a member",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found"),
            }
    )
    @PutMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/folders/{newFolderId}",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputInstitutionalPageDto updateFolderId(
            @PathVariable("institutionalPageId") UUID institutionalPageId,
            @PathVariable("newFolderId") UUID newFolderId
    );

    /**
     * Request update for a specific InstitutionalPage by id, only if the user is a member
     *
     * @param institutionalPageId the id of the InstitutionalPage to update
     * @return the updated {@link OutputRequestUpdateDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Request update for a specific InstitutionalPage by id, only if the user is a member",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found"),
            }
    )
    @PutMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/request-update",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputRequestUpdateDto requestUpdateInstitutionalPageById(
            @PathVariable("institutionalPageId") UUID institutionalPageId
    );

    /**
     * Release lock-for-update of a specific InstitutionalPage by id, only if the user is a member.
     *
     * @param institutionalPageId the id of the InstitutionalPage to update
     */
    @Operation(
            summary = "Release lock-for-update of a specific InstitutionalPage by id, only if the user is a member",
            responses = {
                    @ApiResponse(responseCode = "204", description = "No Content"),
                    @ApiResponse(responseCode = "404", description = "Not Found"),
            }
    )
    @PutMapping(value = "/institutional-page/institutional-pages/{institutionalPageId}/cancel-update")
    @ResponseStatus(HttpStatus.NO_CONTENT)
    void cancelUpdateInstitutionalPageById(
            @PathVariable("institutionalPageId") UUID institutionalPageId
    );

}
