package com.finconsgroup.itserr.marketplace.institutionalpage.dm.repository;

import com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity.InstitutionalPageEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.enums.ModerationStatus;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.util.List;
import java.util.Optional;
import java.util.UUID;

/**
 * {@link InstitutionalPageRepository} repository.
 */
@Repository
public interface InstitutionalPageRepository extends
        JpaRepository<InstitutionalPageEntity, UUID>, JpaSpecificationExecutor<InstitutionalPageEntity> {

    Optional<InstitutionalPageEntity> findByOriginalInstitutionalPageId(UUID originalInstitutionalPageIds);

    List<InstitutionalPageEntity> findAllByParentInstitutionalPageIdInAndModerationStatusIn(
            List<UUID> parentInstitutionalPageIds,
            List<ModerationStatus> moderationStatuses
    );

    List<InstitutionalPageEntity> findAllByParentInstitutionalPageIdAndOriginalInstitutionalPageIdIsNull(
            UUID parentInstitutionalPageId
    );

    @Query(
            value = """
                    select ip.*
                    from institutional_page ip
                    where
                        (:applyModerationStatusesFilter = false or ip.moderation_status in (:moderationStatuses))
                        and (:applyInstitutionalPageIdsFilter = false or coalesce(ip.original_institutional_page_id, ip.id) in (:institutionalPageIds))
                        and (:published is null or ip.published = :published)
                        and (:category is null or ip.category = :category)
                        and (:rootLevelOnly = false or ip.parent_institutional_page_id is null)
                        and (:originalVersionOnly = false or ip.original_institutional_page_id is null)
                        and (:searchText is null or lower(ip.name) like lower(concat('%', cast(:searchText as varchar), '%')))
                    """,
            countQuery = """
                    select count(ip.id)
                    from institutional_page ip
                    where
                        (:applyModerationStatusesFilter = false or ip.moderation_status in (:moderationStatuses))
                        and (:applyInstitutionalPageIdsFilter = false or coalesce(ip.original_institutional_page_id, ip.id) in (:institutionalPageIds))
                        and (:published is null or ip.published = :published)
                        and (:category is null or ip.category = :category)
                        and (:rootLevelOnly = false or ip.parent_institutional_page_id is null)
                        and (:originalVersionOnly = false or ip.original_institutional_page_id is null)
                        and (:searchText is null or lower(ip.name) like lower(concat('%', cast(:searchText as varchar), '%')))
                    """,
            nativeQuery = true
    )
    Page<InstitutionalPageEntity> findAllFiltered(
            Pageable pageable,
            @Param("applyModerationStatusesFilter") boolean applyModerationStatusesFilter,
            @Param("moderationStatuses") List<String> moderationStatuses,
            @Param("applyInstitutionalPageIdsFilter") boolean applyInstitutionalPageIdsFilter,
            @Param("institutionalPageIds") List<UUID> institutionalPageIds,
            @Param("published") Boolean published,
            @Param("category") String category,
            @Param("rootLevelOnly") boolean rootLevelOnly,
            @Param("originalVersionOnly") boolean originalVersionOnly,
            @Param("searchText") String searchText
    );

    @Query(
            value = """
                    WITH member_pages AS (
                        SELECT ip.*
                        FROM institutional_page ip
                        JOIN institutional_page_member member
                            ON (
                                member.member_id = :userId
                                AND (:wpLeaderOnly = false OR member.wp_lead = true)
                                AND (
                                    (
                                        member.wp_lead = false
                                        AND
                                        coalesce(ip.original_institutional_page_id, ip.id) = member.institutional_page_id
                                    )
                                    OR
                                    (
                                        member.wp_lead = true
                                        AND
                                        coalesce(ip.ancestor_institutional_page_ids[1], ip.original_institutional_page_id, ip.id) = member.institutional_page_id
                                    )
                                )
                            )
                        WHERE
                            (:applyModerationStatusesFilter = false OR ip.moderation_status IN (:moderationStatuses))
                            AND (:applyInstitutionalPageIdsFilter = false OR coalesce(ip.original_institutional_page_id, ip.id) IN (:institutionalPageIds))
                            AND (:category IS NULL OR ip.category = :category)
                            AND (:rootLevelOnly = false OR ip.parent_institutional_page_id IS NULL)
                            AND (:originalVersionOnly = false OR ip.original_institutional_page_id IS NULL)
                            AND (:searchText IS NULL or lower(ip.name) LIKE lower(concat('%', cast(:searchText AS varchar), '%')))
                    ),
                    published_pages AS (
                        SELECT ip.*
                        FROM institutional_page ip
                        WHERE ip.published = true
                            AND (:applyModerationStatusesFilter = false OR ip.moderation_status IN (:moderationStatuses))
                            AND (:applyInstitutionalPageIdsFilter = false OR coalesce(ip.original_institutional_page_id, ip.id) IN (:institutionalPageIds))
                            AND (:category IS NULL OR ip.category = :category)
                            AND (:rootLevelOnly = false OR ip.parent_institutional_page_id IS NULL)
                            AND (:originalVersionOnly = false OR ip.original_institutional_page_id IS NULL)
                            AND (:searchText IS NULL or lower(ip.name) LIKE lower(concat('%', cast(:searchText AS varchar), '%')))
                    ),
                    published_member_pages AS (
                        SELECT * FROM member_pages
                        INTERSECT
                        SELECT * FROM published_pages
                    ),
                    published_not_member_pages AS (
                        SELECT * FROM published_pages
                        EXCEPT
                        SELECT * FROM member_pages
                    ),
                    not_published_member_pages AS (
                        SELECT * FROM member_pages
                        WHERE published = false
                    ),
                    result_pages AS (
                        SELECT *
                        FROM (
                            SELECT * FROM not_published_member_pages
                            WHERE :includePrivateAndMember = true
                    
                            UNION ALL
                    
                            SELECT * FROM published_member_pages
                            WHERE :includePublishedAndMember = true
                    
                            UNION ALL
                    
                            SELECT * FROM published_not_member_pages
                            WHERE :includePublishedAndNotMember = true
                        ) AS combined_results
                    )
                    
                    SELECT * FROM result_pages
                    """,
            countQuery = """
                    WITH member_pages AS (
                        SELECT ip.*
                        FROM institutional_page ip
                        JOIN institutional_page_member member
                            ON (
                                member.member_id = :userId
                                AND (:wpLeaderOnly = false OR member.wp_lead = true)
                                AND (
                                    (
                                        member.wp_lead = false
                                        AND
                                        coalesce(ip.original_institutional_page_id, ip.id) = member.institutional_page_id
                                    )
                                    OR
                                    (
                                        member.wp_lead = true
                                        AND
                                        coalesce(ip.ancestor_institutional_page_ids[1], ip.original_institutional_page_id, ip.id) = member.institutional_page_id
                                    )
                                )
                            )
                        WHERE
                            (:applyModerationStatusesFilter = false OR ip.moderation_status IN (:moderationStatuses))
                            AND (:applyInstitutionalPageIdsFilter = false OR coalesce(ip.original_institutional_page_id, ip.id) IN (:institutionalPageIds))
                            AND (:category IS NULL OR ip.category = :category)
                            AND (:rootLevelOnly = false OR ip.parent_institutional_page_id IS NULL)
                            AND (:originalVersionOnly = false OR ip.original_institutional_page_id IS NULL)
                            AND (:searchText IS NULL or lower(ip.name) LIKE lower(concat('%', cast(:searchText AS varchar), '%')))
                    ),
                    published_pages AS (
                        SELECT ip.*
                        FROM institutional_page ip
                        WHERE ip.published = true
                            AND (:applyModerationStatusesFilter = false OR ip.moderation_status IN (:moderationStatuses))
                            AND (:applyInstitutionalPageIdsFilter = false OR coalesce(ip.original_institutional_page_id, ip.id) IN (:institutionalPageIds))
                            AND (:category IS NULL OR ip.category = :category)
                            AND (:rootLevelOnly = false OR ip.parent_institutional_page_id IS NULL)
                            AND (:originalVersionOnly = false OR ip.original_institutional_page_id IS NULL)
                            AND (:searchText IS NULL or lower(ip.name) LIKE lower(concat('%', cast(:searchText AS varchar), '%')))
                    ),
                    published_member_pages AS (
                        SELECT * FROM member_pages
                        INTERSECT
                        SELECT * FROM published_pages
                    ),
                    published_not_member_pages AS (
                        SELECT * FROM published_pages
                        EXCEPT
                        SELECT * FROM member_pages
                    ),
                    not_published_member_pages AS (
                        SELECT * FROM member_pages
                        WHERE published = false
                    ),
                    result_pages AS (
                        SELECT *
                        FROM (
                            SELECT * FROM not_published_member_pages
                            WHERE :includePrivateAndMember = true
                    
                            UNION ALL
                    
                            SELECT * FROM published_member_pages
                            WHERE :includePublishedAndMember = true
                    
                            UNION ALL
                    
                            SELECT * FROM published_not_member_pages
                            WHERE :includePublishedAndNotMember = true
                        ) AS combined_results
                    )
                    
                    SELECT COUNT(id) FROM result_pages
                    """,
            nativeQuery = true
    )
    Page<InstitutionalPageEntity> findAllForMemberFiltered(
            Pageable pageable,
            @Param("userId") UUID userId,
            @Param("wpLeaderOnly") boolean wpLeaderOnly,
            @Param("applyModerationStatusesFilter") boolean applyModerationStatusesFilter,
            @Param("moderationStatuses") List<String> moderationStatuses,
            @Param("applyInstitutionalPageIdsFilter") boolean applyInstitutionalPageIdsFilter,
            @Param("institutionalPageIds") List<UUID> institutionalPageIds,
            @Param("includePrivateAndMember") boolean includePrivateAndMember,
            @Param("includePublishedAndMember") boolean includePublishedAndMember,
            @Param("includePublishedAndNotMember") boolean includePublishedAndNotMember,
            @Param("category") String category,
            @Param("rootLevelOnly") boolean rootLevelOnly,
            @Param("originalVersionOnly") boolean originalVersionOnly,
            @Param("searchText") String searchText
    );

    @Query(
            value = """
                    select ip.*
                    from institutional_page ip
                    join institutional_page_member member
                        on
                        (
                            member.member_id = :userId
                            and member.institutional_page_id = :rootInstitutionalPageId
                            and member.wp_lead = true
                            and coalesce(ip.ancestor_institutional_page_ids[1], ip.original_institutional_page_id, ip.id) = member.institutional_page_id
                        )
                    where
                        (:applyModerationStatusesFilter = false or ip.moderation_status in (:moderationStatuses))
                        and (:originalVersionOnly = false or ip.original_institutional_page_id is null)
                    """,
            countQuery = """
                    select count(ip.id)
                    from institutional_page ip
                    join institutional_page_member member
                        on
                        (
                            member.member_id = :userId
                            and member.institutional_page_id = :rootInstitutionalPageId
                            and member.wp_lead = true
                            and coalesce(ip.ancestor_institutional_page_ids[1], ip.original_institutional_page_id, ip.id) = member.institutional_page_id
                        )
                    where
                        (:applyModerationStatusesFilter = false or ip.moderation_status in (:moderationStatuses))
                        and (:originalVersionOnly = false or ip.original_institutional_page_id is null)
                    """,
            nativeQuery = true
    )
    Page<InstitutionalPageEntity> findHierarchyForMemberFiltered(
            Pageable pageable,
            @Param("userId") UUID userId,
            @Param("rootInstitutionalPageId") UUID rootInstitutionalPageId,
            @Param("applyModerationStatusesFilter") boolean applyModerationStatusesFilter,
            @Param("moderationStatuses") List<String> moderationStatuses,
            @Param("originalVersionOnly") boolean originalVersionOnly
    );

    @Query(
            value = """
                    select ip.*
                    from institutional_page ip
                    where
                        coalesce(ip.ancestor_institutional_page_ids[1], ip.original_institutional_page_id, ip.id) = :rootInstitutionalPageId
                        and (:applyModerationStatusesFilter = false or ip.moderation_status in (:moderationStatuses))
                        and (:originalVersionOnly = false or ip.original_institutional_page_id is null)
                    """,
            countQuery = """
                    select count(ip.id)
                    from institutional_page ip
                    where
                        coalesce(ip.ancestor_institutional_page_ids[1], ip.original_institutional_page_id, ip.id) = :rootInstitutionalPageId
                        and (:applyModerationStatusesFilter = false or ip.moderation_status in (:moderationStatuses))
                        and (:originalVersionOnly = false or ip.original_institutional_page_id is null)
                    """,
            nativeQuery = true
    )
    Page<InstitutionalPageEntity> findHierarchyFiltered(
            Pageable pageable,
            @Param("rootInstitutionalPageId") UUID rootInstitutionalPageId,
            @Param("applyModerationStatusesFilter") boolean applyModerationStatusesFilter,
            @Param("moderationStatuses") List<String> moderationStatuses,
            @Param("originalVersionOnly") boolean originalVersionOnly
    );

}
