package com.finconsgroup.itserr.marketplace.institutionalpage.dm.service;

import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputParagraphDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputInstitutionalPageDto;
import org.springframework.lang.NonNull;

import java.util.UUID;

/**
 * Service for handling business logic related to Paragraphs of InstitutionalPages.
 * <p>
 * This interface provides methods to add, update,
 * and remove paragraphs as a member of a specific InstitutionalPage.
 * </p>
 *
 * <p>Example usage:
 * <pre>
 * OutputInstitutionalPageDto outputInstitutionalPageDto = paragraphService.
 *          addParagraph(userId, institutionalPageId, position, inputParagraphDto);
 * </pre>
 * </p>
 */
public interface ParagraphService {

    /**
     * Adds a new paragraph at the specified position to an institutional page.
     *
     * @param userId              the unique identifier of the authenticated user
     * @param institutionalPageId the id of the InstitutionalPage
     * @param position            the position to add the paragraph
     * @param inputParagraphDto   the input DTO containing  paragraph details
     * @return a {@link OutputInstitutionalPageDto} representing the updated InstitutionalPage
     */
    @NonNull
    OutputInstitutionalPageDto addParagraph(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            Integer position,
            @NonNull InputParagraphDto inputParagraphDto
    );

    /**
     * Updates a paragraph of an institutional page.
     *
     * @param userId              the unique identifier of the authenticated user
     * @param institutionalPageId the id of the InstitutionalPage
     * @param paragraphId         the id of the paragraph to update
     * @param inputParagraphDto   the input DTO containing  paragraph details
     * @return a {@link OutputInstitutionalPageDto} representing the updated InstitutionalPage
     */
    @NonNull
    OutputInstitutionalPageDto updateParagraph(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            @NonNull UUID paragraphId,
            @NonNull InputParagraphDto inputParagraphDto
    );

    /**
     * Removes a paragraph from an institutional page.
     *
     * @param userId              the unique identifier of the authenticated user
     * @param institutionalPageId the id of the InstitutionalPage
     * @param paragraphId         the id of the paragraph to remove
     * @return a {@link OutputInstitutionalPageDto} representing the updated InstitutionalPage
     */
    @NonNull
    OutputInstitutionalPageDto removeParagraph(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            @NonNull UUID paragraphId
    );

}
