package com.finconsgroup.itserr.marketplace.institutionalpage.dm.repository;

import com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity.MemberEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.model.MembershipModel;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.util.List;
import java.util.UUID;

/**
 * {@link MemberRepository} repository.
 */
@Repository
public interface MemberRepository extends JpaRepository<MemberEntity, UUID> {

    boolean existsByMemberIdAndInstitutionalPageIdAndWpLead(
            UUID memberId,
            UUID institutionalPageId,
            boolean wpLead
    );

    List<MemberEntity> findAllByInstitutionalPageIdInAndWpLead(
            List<UUID> institutionalPageIds,
            boolean wpLead
    );

    List<MemberEntity> findAllByInstitutionalPageIdAndMemberIdInAndWpLead(
            UUID institutionalPageId,
            List<UUID> memberIds,
            boolean wpLead
    );

    @Modifying
    @Query(
            value = """
                    delete from institutional_page_member member
                    using institutional_page ip
                    where
                        member.institutional_page_id = ip.id
                        and ip.ancestor_institutional_page_ids[1] = :rootInstitutionalPageId
                        and member.wp_lead = false
                        and member.member_id in (:userIds)
                    """,
            nativeQuery = true
    )
    void deleteRegularMembersFromDescendants(
            @Param("rootInstitutionalPageId") UUID rootInstitutionalPageId,
            @Param("userIds") List<UUID> userIds
    );

    @Query(
            value = """
                    select distinct member.member_id, member.wp_lead
                    from institutional_page_member member
                    join institutional_page ip on ip.id = member.institutional_page_id
                    where coalesce(ip.ancestor_institutional_page_ids[1], ip.id) = :rootInstitutionalPageId
                    order by member.wp_lead desc, member.member_id
                    """,
            countQuery = """
                    select count(distinct member.member_id, member.wp_lead)
                    from institutional_page_member member
                    join institutional_page ip on ip.id = member.institutional_page_id
                    where coalesce(ip.ancestor_institutional_page_ids[1], ip.id) = :rootInstitutionalPageId
                    order by member.wp_lead desc, member.member_id
                    """,
            nativeQuery = true
    )
    Page<MembershipModel> findDistinctMemberIdsOrderedByWpLead(
            Pageable pageable,
            @Param("rootInstitutionalPageId") UUID rootInstitutionalPageId
    );

    @Query(
            value = """
                    select member.*
                    from institutional_page_member member
                    join institutional_page ip on ip.id = member.institutional_page_id
                    where
                        coalesce(ip.ancestor_institutional_page_ids[1], ip.id) = :rootInstitutionalPageId
                        and member.member_id in (:memberIds)
                    """,
            nativeQuery = true
    )
    List<MemberEntity> findAllMembersByRootIdAndMemberIds(
            @Param("rootInstitutionalPageId") UUID rootInstitutionalPageId,
            @Param("memberIds") List<UUID> memberIds
    );

}
