/**
 * Global exception handler for REST controllers in the institutional page system.
 * This class handles custom exceptions related to institutional page e member operations
 * and returns appropriate HTTP error responses.
 *
 * <p>Example usage:
 * <pre>
 * When a InstitutionalPageNotFoundException is thrown, this handler returns a 404 NOT FOUND response.
 * </pre>
 * </p>
 */
package com.finconsgroup.itserr.marketplace.institutionalpage.dm.exception;

import com.finconsgroup.itserr.marketplace.core.web.exception.ErrorResponseDto;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.context.request.WebRequest;

import static com.finconsgroup.itserr.marketplace.core.web.exception.ExceptionUtils.createResponseEntityError;
import static com.finconsgroup.itserr.marketplace.core.web.exception.ExceptionUtils.logError;
import static org.springframework.http.HttpStatus.BAD_REQUEST;
import static org.springframework.http.HttpStatus.CONFLICT;
import static org.springframework.http.HttpStatus.FORBIDDEN;
import static org.springframework.http.HttpStatus.NOT_FOUND;

/**
 * Custom Exception Handler.
 */
@RestControllerAdvice
public class CustomExceptionHandler {

    /**
     * Handles InstitutionalPageNotFoundException and returns a 404 NOT FOUND error response.
     *
     * @param ex      the InstitutionalPageNotFoundException that was thrown
     * @param request the current web request
     * @return a ResponseEntity containing the error response DTO and HTTP status 404
     */
    @ExceptionHandler({InstitutionalPageNotFoundException.class})
    public ResponseEntity<ErrorResponseDto> handleInstitutionalPageNotFoundException(
            final InstitutionalPageNotFoundException ex, final WebRequest request
    ) {
        logError(ex, request);
        return createResponseEntityError(NOT_FOUND, ex.getMessage());
    }


    /**
     * Handles ParagraphNotFoundException and returns a 404 NOT FOUND error response.
     *
     * @param ex      the ParagraphNotFoundException that was thrown
     * @param request the current web request
     * @return a ResponseEntity containing the error response DTO and HTTP status 404
     */
    @ExceptionHandler({ParagraphNotFoundException.class})
    public ResponseEntity<ErrorResponseDto> handleParagraphNotFoundException(
            final ParagraphNotFoundException ex, final WebRequest request
    ) {
        logError(ex, request);
        return createResponseEntityError(NOT_FOUND, ex.getMessage());
    }

    /**
     * Handles InstitutionalPageHasChildrenException and returns a 409 CONFLICT error response.
     *
     * @param ex      the InstitutionalPageHasChildrenException that was thrown
     * @param request the current web request
     * @return a ResponseEntity containing the error response DTO and HTTP status 409
     */
    @ExceptionHandler({InstitutionalPageHasChildrenException.class})
    public ResponseEntity<ErrorResponseDto> handleInstitutionalPageHasChildrenException(
            final InstitutionalPageHasChildrenException ex, final WebRequest request
    ) {
        logError(ex, request);
        return createResponseEntityError(CONFLICT, ex.getMessage());
    }

    /**
     * Handles MembersAlreadyWpLeaderInHierarchyException and returns a 409 CONFLICT error response.
     *
     * @param ex      the MembersAlreadyWpLeaderInHierarchyException that was thrown
     * @param request the current web request
     * @return a ResponseEntity containing the error response DTO and HTTP status 409
     */
    @ExceptionHandler({MembersAlreadyWpLeaderInHierarchyException.class})
    public ResponseEntity<ErrorResponseDto> handleMembersAlreadyWpLeaderInHierarchyException(
            final MembersAlreadyWpLeaderInHierarchyException ex, final WebRequest request
    ) {
        logError(ex, request);
        return createResponseEntityError(CONFLICT, ex.getMessage());
    }

    /**
     * Handles NotInstitutionalPageWPLeaderException and returns a 403 FORBIDDEN error response.
     *
     * @param ex      the NotInstitutionalPageWPLeaderException that was thrown
     * @param request the current web request
     * @return a ResponseEntity containing the error response DTO and HTTP status 403
     */
    @ExceptionHandler({NotInstitutionalPageWPLeaderException.class})
    public ResponseEntity<ErrorResponseDto> handleNotInstitutionalPageWPLeaderException(
            final NotInstitutionalPageWPLeaderException ex, final WebRequest request
    ) {
        logError(ex, request);
        return createResponseEntityError(FORBIDDEN, ex.getMessage());
    }

    /**
     * Handles NotInstitutionalPageMemberException and returns a 403 FORBIDDEN error response.
     *
     * @param ex      the NotInstitutionalPageMemberException that was thrown
     * @param request the current web request
     * @return a ResponseEntity containing the error response DTO and HTTP status 403
     */
    @ExceptionHandler({NotInstitutionalPageMemberException.class})
    public ResponseEntity<ErrorResponseDto> handleNotInstitutionalPageMemberException(
            final NotInstitutionalPageMemberException ex, final WebRequest request
    ) {
        logError(ex, request);
        return createResponseEntityError(FORBIDDEN, ex.getMessage());
    }

    /**
     * Handles MemberRequestsNotFoundException and returns a 404 NOT FOUND error response.
     *
     * @param ex      the MemberRequestsNotFoundException that was thrown
     * @param request the current web request
     * @return a ResponseEntity containing the error response DTO and HTTP status 404
     */
    @ExceptionHandler({MemberRequestsNotFoundException.class})
    public ResponseEntity<ErrorResponseDto> handleMemberRequestsNotFoundException(
            final MemberRequestsNotFoundException ex, final WebRequest request
    ) {
        logError(ex, request);
        return createResponseEntityError(NOT_FOUND, ex.getMessage());
    }

    /**
     * Handles MembersNotFoundException and returns a 404 NOT FOUND error response.
     *
     * @param ex      the MembersNotFoundException that was thrown
     * @param request the current web request
     * @return a ResponseEntity containing the error response DTO and HTTP status 404
     */
    @ExceptionHandler({MembersNotFoundException.class})
    public ResponseEntity<ErrorResponseDto> handleMembersNotFoundException(
            final MembersNotFoundException ex, final WebRequest request
    ) {
        logError(ex, request);
        return createResponseEntityError(NOT_FOUND, ex.getMessage());
    }

    /**
     * Handles InvalidParagraphPositionException and returns a 400 BAD REQUEST error response.
     *
     * @param ex      the InvalidParagraphPositionException that was thrown
     * @param request the current web request
     * @return a ResponseEntity containing the error response DTO and HTTP status 400
     */
    @ExceptionHandler({InvalidParagraphPositionException.class})
    public ResponseEntity<ErrorResponseDto> handleInvalidParagraphPositionException(
            final InvalidParagraphPositionException ex, final WebRequest request
    ) {
        logError(ex, request);
        return createResponseEntityError(BAD_REQUEST, ex.getMessage());
    }

    /**
     * Handles InvalidModerationStatusException and returns a 409 CONFLICT error response.
     *
     * @param ex      the InvalidModerationStatusException that was thrown
     * @param request the current web request
     * @return a ResponseEntity containing the error response DTO and HTTP status 409
     */
    @ExceptionHandler({InvalidModerationStatusException.class})
    public ResponseEntity<ErrorResponseDto> handleInvalidModerationStatusException(
            final InvalidModerationStatusException ex, final WebRequest request
    ) {
        logError(ex, request);
        return createResponseEntityError(CONFLICT, ex.getMessage());
    }

    /**
     * Handles UsersAlreadyMemberException and returns a 409 CONFLICT error response.
     *
     * @param ex      the UsersAlreadyMemberException that was thrown
     * @param request the current web request
     * @return a ResponseEntity containing the error response DTO and HTTP status 409
     */
    @ExceptionHandler({UsersAlreadyMemberException.class})
    public ResponseEntity<ErrorResponseDto> handleUserAlreadyMemberException(
            final UsersAlreadyMemberException ex, final WebRequest request
    ) {
        logError(ex, request);
        return createResponseEntityError(CONFLICT, ex.getMessage());
    }

    /**
     * Handles MemberRequestsAlreadyExistException and returns a 409 CONFLICT error response.
     *
     * @param ex      the MemberRequestsAlreadyExistException that was thrown
     * @param request the current web request
     * @return a ResponseEntity containing the error response DTO and HTTP status 409
     */
    @ExceptionHandler({MemberRequestsAlreadyExistException.class})
    public ResponseEntity<ErrorResponseDto> handleMemberRequestsAlreadyExistException(
            final MemberRequestsAlreadyExistException ex, final WebRequest request
    ) {
        logError(ex, request);
        return createResponseEntityError(CONFLICT, ex.getMessage());
    }

    /**
     * Handles UnexpectedModerationStatusException and returns a 409 CONFLICT error response.
     *
     * @param ex      the UnexpectedModerationStatusException that was thrown
     * @param request the current web request
     * @return a ResponseEntity containing the error response DTO and HTTP status 409
     */
    @ExceptionHandler({UnexpectedModerationStatusException.class})
    public ResponseEntity<ErrorResponseDto> handleUnexpectedModerationStatusException(
            final UnexpectedModerationStatusException ex, final WebRequest request
    ) {
        logError(ex, request);
        return createResponseEntityError(CONFLICT, ex.getMessage());
    }

    /**
     * Handles CannotRemoveMaintainerException and returns a 409 CONFLICT error response.
     *
     * @param ex      the CannotRemoveMaintainerException that was thrown
     * @param request the current web request
     * @return a ResponseEntity containing the error response DTO and HTTP status 409
     */
    @ExceptionHandler({CannotRemoveMaintainerException.class})
    public ResponseEntity<ErrorResponseDto> handleCannotRemoveMaintainerException(
            final CannotRemoveMaintainerException ex, final WebRequest request
    ) {
        logError(ex, request);
        return createResponseEntityError(CONFLICT, ex.getMessage());
    }

    /**
     * Handles TargetUserNotMemberException and returns a 409 CONFLICT error response.
     *
     * @param ex      the TargetUserNotMemberException that was thrown
     * @param request the current web request
     * @return a ResponseEntity containing the error response DTO and HTTP status 409
     */
    @ExceptionHandler({TargetUserNotMemberException.class})
    public ResponseEntity<ErrorResponseDto> handleTargetUserNotMemberException(
            final TargetUserNotMemberException ex, final WebRequest request
    ) {
        logError(ex, request);
        return createResponseEntityError(CONFLICT, ex.getMessage());
    }

    /**
     * Handles UpdateLockedException and returns a 409 CONFLICT error response.
     *
     * @param ex      the UpdateLockedException that was thrown
     * @param request the current web request
     * @return a ResponseEntity containing the error response DTO and HTTP status 409
     */
    @ExceptionHandler({UpdateLockedException.class})
    public ResponseEntity<ErrorResponseDto> handleUpdateLockedException(
            final UpdateLockedException ex, final WebRequest request
    ) {
        logError(ex, request);
        return createResponseEntityError(CONFLICT, ex.getMessage());
    }

}
