package com.finconsgroup.itserr.marketplace.metadata.bs.service.impl;

import com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm.MetadataDmClient;
import com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm.dto.InputCreateMetadataDmDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm.dto.OutputMetadataDmDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.InputCreateMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.OutputMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.exception.MetadataExistsException;
import com.finconsgroup.itserr.marketplace.metadata.bs.exception.WP2ServiceUnavailableException;
import com.finconsgroup.itserr.marketplace.metadata.bs.mapper.MetadataMapper;
import com.finconsgroup.itserr.marketplace.metadata.bs.messaging.ResourceProducer;
import com.finconsgroup.itserr.marketplace.metadata.bs.service.MetadataService;
import feign.FeignException;
import lombok.RequiredArgsConstructor;
import org.jetbrains.annotations.NotNull;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;

import java.util.UUID;

/**
 * Default implementation of a {@link MetadataService}.
 */
@Service
@RequiredArgsConstructor
public class DefaultMetadataService implements MetadataService {

    /** Metadata mapper. */
    private final MetadataMapper metadataMapper;

    private final MetadataDmClient metadataDmClient;

    private final ResourceProducer resourceProducer;

    @Override
    @NonNull
    public OutputMetadataDto create(@NotNull final InputCreateMetadataDto request) {

        // Create metadata
        final OutputMetadataDto createdMetadata = executeCreateMetadata(request);

        // Send message into bus
        resourceProducer.publishCreatedResource(createdMetadata);

        // Return created metadata
        return createdMetadata;

    }

    /**
     * Creates a metadata calling the DM service.
     * @param request Request.
     * @return Created metadata.
     * @throws com.finconsgroup.itserr.marketplace.metadata.bs.exception.MetadataExistsException when a metadata with the same name already exists.
     * @throws WP2ServiceUnavailableException when downstream service is not available.
     */
    private OutputMetadataDto executeCreateMetadata(@NotNull InputCreateMetadataDto request) {

        // Map to client request
        final InputCreateMetadataDmDto clientRequest = metadataMapper.dtoToClientDto(request);

        // Execute
        final OutputMetadataDmDto createdMetadata;
        try {

            // Create metadata using remote service
            createdMetadata = metadataDmClient.create(clientRequest);

        } catch (final FeignException.Conflict e) {

            // Another metadata exists with the same name
            throw new MetadataExistsException(request.getName());

        }

        // Return created metadata
        return metadataMapper.clientDtoToDto(createdMetadata);

    }

    @NotNull
    @Override
    public OutputMetadataDto deleteById(@NotNull final UUID id) {

        // Delete metadata
        final OutputMetadataDto deletedMetadata = metadataMapper.clientDtoToDto(
                metadataDmClient.delete(id));

        // Send message into bus
        resourceProducer.publishDeletedResource(deletedMetadata.getId());

        // Return deleted metadata
        return deletedMetadata;

    }

}
