package com.finconsgroup.itserr.marketplace.metadata.bs.dto;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonValue;
import io.swagger.v3.oas.annotations.media.Schema;
import org.jetbrains.annotations.NotNull;
import org.springframework.core.convert.converter.Converter;
import org.springframework.stereotype.Component;

/**
 * Enumeration representing the possible statuses of a metadata.
 */
@Schema(description = "Enumeration representing the possible statuses of a metadata.")
public enum MetadataStatus {
    /**
     * Metadata is in draft state, being worked on by the creator.
     */
    DRAFT("draft"),

    /**
     * Metadata has been submitted and is pending moderation review.
     */
    PENDING("pending"),

    /**
     * Metadata has been approved by moderators.
     */
    APPROVED("approved"),

    /**
     * Metadata has been rejected by moderators.
     */
    REJECTED("rejected");

    private final String value;

    MetadataStatus(String value) {
        this.value = value;
    }

    @JsonValue
    public String getValue() {
        return value;
    }

    @Override
    public String toString() {
        return value;
    }

    @JsonCreator
    public static MetadataStatus fromString(String value) {
        if (value == null) {
            return null;
        }
        for (MetadataStatus status : MetadataStatus.values()) {
            if (status.value.equalsIgnoreCase(value)) {
                return status;
            }
        }
        throw new IllegalArgumentException("Unknown MetadataStatus: " + value);
    }

    @Component
    public static class MetadataStatusConverter implements Converter<String, MetadataStatus> {
        @Override
        public MetadataStatus convert(@NotNull String source) {
            return MetadataStatus.fromString(source);
        }
    }
}