package com.finconsgroup.itserr.marketplace.metadata.dm.controller;

import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.metadata.dm.api.MetadataApi;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.InputCreateMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.OutputMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.OutputMetadataFieldDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.OutputMetadataFieldExtDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.OutputMetadataPreviewDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.entity.enumerated.MetadataCategoryEnum;
import com.finconsgroup.itserr.marketplace.metadata.dm.service.MetadataService;

import jakarta.validation.constraints.Positive;
import jakarta.validation.constraints.PositiveOrZero;
import lombok.RequiredArgsConstructor;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort.Direction;
import org.springframework.web.bind.annotation.RestController;

import java.util.UUID;

/**
 * REST controller class for handling Metadata related API requests.
 *
 * <p>Implements the {@link MetadataApi} interface.</p>
 */
@RequiredArgsConstructor
@RestController
public class MetadataController implements MetadataApi {

    private final MetadataService metadataService;

    @Override
    public Page<OutputMetadataPreviewDto> findAll(
            MetadataCategoryEnum category,
            final int pageNumber,
            final int pageSize,
            final String sort,
            final Direction direction
    ) {
        Pageable pageable = PageRequest.of(pageNumber, pageSize, direction, sort);
        return metadataService.findAll(category, pageable);
    }

    @Override
    public OutputMetadataDto findById(final UUID metadataId) {
        return this.metadataService.findById(metadataId);
    }

    @Override
    public OutputMetadataDto create(final InputCreateMetadataDto request) {
        return metadataService.create(request, JwtTokenHolder.getUserIdOrThrow());
    }

    @Override
    public OutputMetadataDto deleteById(final UUID metadataId) {
        JwtTokenHolder.getUserIdOrThrow();
        return this.metadataService.deleteById(metadataId);
    }

    @Override
    public Page<OutputMetadataFieldExtDto> findAllFields(MetadataCategoryEnum category, @PositiveOrZero int pageNumber,
                                                         @Positive int pageSize, String sort, Direction direction) {
        Pageable pageable = PageRequest.of(pageNumber, pageSize, direction, sort);
        return metadataService.findAllFields(category, pageable);
        
    }

    @Override
    public Page<OutputMetadataFieldDto> findAllFieldsById(
            UUID metadataId,
            final int pageNumber,
            final int pageSize,
            final String sort,
            final Direction direction) {
        Pageable pageable = PageRequest.of(pageNumber, pageSize, direction, sort);
        return metadataService.findAllFieldsById(metadataId, pageable);
    }
}
