package com.finconsgroup.itserr.marketplace.metadata.dm.repository.specification;

import com.finconsgroup.itserr.marketplace.metadata.dm.entity.MetadataEntity;
import jakarta.persistence.criteria.CriteriaBuilder;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;

import java.util.Collection;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * {@link MetadataEntity} specifications.
 */
public final class MetadataSpecifications {

    /**
     * Private constructor.
     */
    private MetadataSpecifications() {
        throw new UnsupportedOperationException("Cannot be instantiated");
    }

    /**
     * <p>Creates a {@link Specification} to find {@link MetadataEntity} by argument ids.</p>
     * <p>If null or empty list is passed, a {@link CriteriaBuilder#conjunction()} is returned, matching any entity.</p>
     * @param ids Metadata ids, or null.
     * @return A {@link Specification} matching ids.
     */
    @NonNull
    public static Specification<MetadataEntity> ids(@Nullable final Collection<UUID> ids) {
        return (root, query, cb) ->
                sanitizeIds(ids)
                        // TODO: may we use JPA Metamodel?
                        .map(root.get("id")::in)
                        .orElse(cb.conjunction());
    }

    /**
     * <p>Given a collection of ids, returns an {@link Optional} containing a {@link Set} with unique non-null ids.</p>
     * <p>If the argument is null or the resulting Set is empty, an empty Optional is returned.</p>
     * @param ids IDs. May be null.
     * @return Optional containing a Set of unique non-null ids, or empty Optional if the resulting set is empty.
     */
    @NonNull
    private static Optional<Set<UUID>> sanitizeIds(@Nullable final Collection<UUID> ids) {

        // If ids is null, return empty optional
        if (ids == null) {
            return Optional.empty();
        }

        // Convert to Set, excluding null values
        final Set<UUID> sanitizedIds = ids.stream()
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());

        // Return an Optional with the non-empty Set, or an empty Optional if Set is empty
        return sanitizedIds.isEmpty()
                ? Optional.empty()
                : Optional.of(sanitizedIds);

    }

}
