package com.finconsgroup.itserr.marketplace.metrics.dm.bs;

import com.finconsgroup.itserr.marketplace.metrics.dm.config.properties.HibernateConfigurationProperties;
import com.finconsgroup.itserr.marketplace.metrics.dm.exception.UpdateMetricsException;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;
import java.util.Optional;

/**
 * Component responsible for managing and executing updates of materialized views in the database.
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class MVUpdater {

    /** Name of the event daily counts materialized view */
    private static final String MV_EVENT_DAILY_COUNTS_NAME = "metric_event_daily_counts";
    /** Name of the beneficiary event daily counts materialized view */
    private static final String MV_BENEFICIARY_EVENT_DAILY_COUNTS_NAME = "beneficiary_metric_event_daily_counts";
    /** Name of the beneficiary metric interest materialized view */
    private static final String MV_BENEFICIARY_METRIC_INTEREST_NAME = "beneficiary_metric_interest";
    /** Name of the beneficiary interest materialized view */
    private static final String MV_BENEFICIARY_INTEREST_NAME = "beneficiary_interest";

    /** JDBC template for executing database operations */
    private final JdbcTemplate jdbc;

    /** Hibernate configuration properties containing database settings */
    private final HibernateConfigurationProperties config;

    /** Timestamp of the last successful materialized view update */
    @Getter
    private LocalDateTime lastUpdate;

    /**
     * Updates the materialized view in a thread-safe manner.
     *
     * @throws UpdateMetricsException if an error occurs during the update process
     */
    public void update() {
        log.debug("Updating materialized views");
        synchronized (this) {
            final LocalDateTime updateStartTime = LocalDateTime.now();
            final String schemaPrefix = Optional.ofNullable(config.getDefaultSchema())
                    .filter(StringUtils::isNotBlank)
                    .map(d -> "\"" + d + "\".")
                    .orElse("");
            updateMV(schemaPrefix, MV_EVENT_DAILY_COUNTS_NAME);
            updateMV(schemaPrefix, MV_BENEFICIARY_EVENT_DAILY_COUNTS_NAME);
            updateMV(schemaPrefix, MV_BENEFICIARY_METRIC_INTEREST_NAME);
            updateMV(schemaPrefix, MV_BENEFICIARY_INTEREST_NAME);
            lastUpdate = updateStartTime;
        }
        log.debug("Materialized views update completed");
    }

    /**
     * Updates a specific materialized view in the database.
     *
     * @param schemaPrefix The schema prefix to be used for the materialized view, including schema name and delimiter. Can be an empty string if no
     *         schema prefix is required.
     * @param mvName The name of the materialized view to be updated. This value must represent a valid materialized view in the database.
     * @throws UpdateMetricsException if an error occurs while attempting to refresh the materialized view.
     */
    private void updateMV(
            final String schemaPrefix,
            final String mvName) {
        log.debug("Updating materialized view {}", mvName);
        try {
            jdbc.execute("REFRESH MATERIALIZED VIEW CONCURRENTLY " + schemaPrefix + "\"" + mvName + "\"");
        } catch (Exception e) {
            throw new UpdateMetricsException("Error updating materialized view: " + mvName, e);
        }
    }

    /**
     * Updates the materialized view only if it hasn't been updated since the specified time.
     *
     * @param time The timestamp to compare against the last update time
     * @return true if the update was performed, false if skipped due to a recent update
     */
    public boolean updateIfElderThan(final LocalDateTime time) {
        final LocalDateTime localLastUpdate;
        synchronized (this) {
            localLastUpdate = this.lastUpdate;
        }
        if (localLastUpdate == null || localLastUpdate.isBefore(time)) {
            this.update();
            return true;
        } else {
            log.debug("Materialized view {} update aborted (already updated)", MV_EVENT_DAILY_COUNTS_NAME);
            return false;
        }
    }

}
