package com.finconsgroup.itserr.marketplace.news.bs.client;

import com.finconsgroup.itserr.marketplace.news.bs.client.dto.workspace.FolderDto;
import com.finconsgroup.itserr.marketplace.news.bs.client.dto.workspace.InputFolderCreateDto;
import com.finconsgroup.itserr.marketplace.news.bs.client.dto.workspace.OutputPublicImageDto;
import com.finconsgroup.itserr.marketplace.news.bs.client.dto.workspace.WorkspaceDto;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import jakarta.validation.constraints.NotBlank;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RequestPart;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.multipart.MultipartFile;

/**
 * Feign client for communication with the UserProfile DM service.
 */
public interface WorkspaceBsClient {

    /**
     * Retrieve the user's workspace
     */
    @GetMapping(path = "/workspace", produces = MediaType.APPLICATION_JSON_VALUE)
    WorkspaceDto getWorkspace();

    /**
     * Allow the folder creation in a user's workspace.
     *
     * @param folderId The ID of the parent folder.
     * @return a string containing the {folderId} of the newly created folder.
     */
    @PostMapping(
            path = "workspace/folders/{folderId}/folders",
            produces = MediaType.APPLICATION_JSON_VALUE,
            consumes = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.CREATED)
    String createFolderByFolderId(
            @PathVariable @NotBlank String folderId,
            @RequestBody InputFolderCreateDto inputFolderCreateDto);

    /**
     * Retrieves the folder in the parent folder id matching the provided name.
     *
     * @param folderId The ID of the parent folder.
     * @param name     The name of the folder to match.
     * @return A list of {@link FolderDto} representing the folders and sub-folders.
     */
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Folder found"),
            @ApiResponse(responseCode = "400", description = "Bad request"),
            @ApiResponse(responseCode = "404", description = "Folder not found")
    })
    @GetMapping(path = "workspace/folders/{folderId}/folder-by-name", produces = MediaType.APPLICATION_JSON_VALUE)
    FolderDto getFolderByParentFolderIdAndName(
            @PathVariable String folderId,
            @RequestParam String name
    );

    /**
     * Creates an image file, creating a new version of the document.
     *
     * @param folderId Id of the parent folder to create image in.
     * @param file     The new image file.
     * @return Created image.
     */
    @PostMapping(path = "/workspace/images/public/{folderId}", consumes = MediaType.MULTIPART_FORM_DATA_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE)
    OutputPublicImageDto createPublicImage(
            @PathVariable(name = "folderId") String folderId,
            @RequestPart(name = "file") MultipartFile file);

    /**
     * Replaces an image file, creating a new version of the document.
     *
     * @param documentId Id of the image document to be updated.
     * @param file       The new image file.
     * @return Updated image.
     */
    @PutMapping(path = "/workspace/images/public/{documentId}", consumes = MediaType.MULTIPART_FORM_DATA_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE)
    OutputPublicImageDto updatePublicImage(
            @PathVariable(name = "documentId") String documentId,
            @RequestPart(name = "file") MultipartFile file);

}
