package com.finconsgroup.itserr.marketplace.news.bs.service;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.news.bs.client.dto.newsDm.InputCreateNewsDto;
import com.finconsgroup.itserr.marketplace.news.bs.client.dto.newsDm.InputUpdateNewsDto;
import com.finconsgroup.itserr.marketplace.news.bs.dto.OutputNewsDetailDto;
import org.springframework.data.domain.Sort;
import org.springframework.lang.NonNull;

import java.util.UUID;

/**
 * Service for handling business logic related to News. This interface provides methods to retrieve, create, update, and delete News,
 *
 * <p>Example usage:
 * <pre>
 * Page<OutputNewsDetailDto> News = NewsService.findAll(pageable);
 * </pre>
 * </p>
 */
public interface NewsService {

    /**
     * Retrieves a paginated list of all News.
     *
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize   the number of UserProfiles per page (default is 10)
     * @param sort       the field to sort by (default is "startTime")
     * @param direction  the direction of sorting (default is descending)
     * @return a page of OutputNewsDetailDto representing the News
     */
    @NonNull
    OutputPageDto<OutputNewsDetailDto> findAll(@NonNull int pageNumber,
                                               @NonNull int pageSize,
                                               @NonNull String sort,
                                               @NonNull Sort.Direction direction);

    /**
     * Retrieves news by id.
     *
     * @param newsId the id of the news
     * @return an OutputNewsDetailDto representing the news
     * @throws WP2ResourceNotFoundException if no news is found with the given id
     */
    @NonNull
    OutputNewsDetailDto findById(@NonNull UUID newsId);

    /**
     * Creates new news for user using the provided input data.
     *
     * @param inputCreateNewsDto the input DTO containing news details
     */
    @NonNull
    OutputNewsDetailDto create(@NonNull InputCreateNewsDto inputCreateNewsDto);

    /**
     * Updates existing news by id with the provided data.
     *
     * @param newsId             the id of the news to update
     * @param inputUpdateNewsDto the input DTO containing updated  news details
     * @return an OutputNewsDetailDto newsDto representing the updated news
     * @throws WP2ResourceNotFoundException if no news is found with the given id
     */
    @NonNull
    OutputNewsDetailDto updateById(@NonNull UUID newsId,
                                   @NonNull InputUpdateNewsDto inputUpdateNewsDto);


    /**
     * Deletes news by id. It will move the news entity on a separated table containing deleted records.
     *
     * @param newsId the id of the news to delete
     * @throws WP2ResourceNotFoundException if no news is found with the given id
     */
    void deleteById(@NonNull UUID newsId);
}
