package com.finconsgroup.itserr.marketplace.news.bs.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;

import com.finconsgroup.itserr.marketplace.news.bs.client.WorkspaceBsClient;
import com.finconsgroup.itserr.marketplace.news.bs.client.dto.newsDm.OutputImageDto;
import com.finconsgroup.itserr.marketplace.news.bs.client.dto.workspace.FolderDto;
import com.finconsgroup.itserr.marketplace.news.bs.client.dto.workspace.InputFolderCreateDto;
import com.finconsgroup.itserr.marketplace.news.bs.client.dto.workspace.OutputPublicImageDto;
import com.finconsgroup.itserr.marketplace.news.bs.client.dto.workspace.WorkspaceDto;
import com.finconsgroup.itserr.marketplace.news.bs.config.properties.NewsBsConfigurationProperties;
import com.finconsgroup.itserr.marketplace.news.bs.service.ImageService;
import feign.FeignException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.util.UUID;

import static com.finconsgroup.itserr.marketplace.news.bs.util.Constants.IMAGE_FOLDER_DESCRIPTION;
import static com.finconsgroup.itserr.marketplace.news.bs.util.Constants.HIDDEN_FOLDER_FLAG;

/**
 * Default implementation of {@link ImageService} to perform operations related to images
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultImageService implements ImageService {

    private final WorkspaceBsClient workspaceBsClient;
    private final NewsBsConfigurationProperties newsBsConfigurationProperties;

    @NonNull
    @Override
    public OutputImageDto createPublicImage(@NonNull MultipartFile file) {
        String folderId = getOrCreateImagesFolder();
        OutputPublicImageDto wsOutputPublicImageWSDto = workspaceBsClient.createPublicImage(folderId, file);
        return mapOutputPublicImageDto(wsOutputPublicImageWSDto);
    }

    @NonNull
    @Override
    public OutputImageDto updatePublicImage(@NonNull String id, @NonNull MultipartFile file) {
        OutputPublicImageDto wsOutputPublicImageWSDto = workspaceBsClient.updatePublicImage(id, file);
        return mapOutputPublicImageDto(wsOutputPublicImageWSDto);
    }

    /*
     * Creates hidden folder to upload images related to news in user's workspace,
     * if it does not already exist.
     */
    private String getOrCreateImagesFolder() {
        WorkspaceDto workspace = workspaceBsClient.getWorkspace();
        try {
            FolderDto folder = workspaceBsClient.getFolderByParentFolderIdAndName(workspace.getId(),
                    newsBsConfigurationProperties.getImagesFolderName());
            return folder.getId();
        } catch (FeignException.NotFound | WP2ResourceNotFoundException e) {
            return workspaceBsClient.createFolderByFolderId(workspace.getId(), getImagesFolderPayload());
        }
    }

    private InputFolderCreateDto getImagesFolderPayload() {
        return InputFolderCreateDto.builder()
                .name(newsBsConfigurationProperties.getImagesFolderName())
                .description(IMAGE_FOLDER_DESCRIPTION)
                .hidden(HIDDEN_FOLDER_FLAG)
                .build();
    }

    /*
     * Maps the OutputPublicImageDto from Workspace BS to Event BS's OutputImageDto
     */
    private static OutputImageDto mapOutputPublicImageDto(OutputPublicImageDto outputPublicImageDto) {
        return OutputImageDto.builder()
                .id(UUID.fromString(outputPublicImageDto.getId()))
                .url(outputPublicImageDto.getPublicLink())
                .build();
    }
}
