package com.finconsgroup.itserr.marketplace.news.bs.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.dto.PageRequestDto;
import com.finconsgroup.itserr.marketplace.news.bs.bean.DetailRequest;
import com.finconsgroup.itserr.marketplace.news.bs.bean.NewsApplicationEvent;
import com.finconsgroup.itserr.marketplace.news.bs.client.NewsDmClient;
import com.finconsgroup.itserr.marketplace.news.bs.client.dto.newsDm.InputCreateNewsDto;
import com.finconsgroup.itserr.marketplace.news.bs.client.dto.newsDm.InputUpdateNewsDto;
import com.finconsgroup.itserr.marketplace.news.bs.client.dto.newsDm.OutputAuthorDto;
import com.finconsgroup.itserr.marketplace.news.bs.client.dto.newsDm.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.news.bs.client.dto.newsDm.OutputNewsDto;
import com.finconsgroup.itserr.marketplace.news.bs.dto.OutputNewsAuthorDto;
import com.finconsgroup.itserr.marketplace.news.bs.dto.OutputNewsDetailDto;
import com.finconsgroup.itserr.marketplace.news.bs.dto.OutputNewsInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.news.bs.service.NewsAuthorDetailProvider;
import com.finconsgroup.itserr.marketplace.news.bs.service.NewsInstitutionalPageProvider;
import com.finconsgroup.itserr.marketplace.news.bs.service.NewsService;
import jakarta.validation.constraints.NotEmpty;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.data.domain.Sort;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.finconsgroup.itserr.marketplace.news.bs.enums.EventType.CREATED;
import static com.finconsgroup.itserr.marketplace.news.bs.enums.EventType.DELETED;
import static com.finconsgroup.itserr.marketplace.news.bs.enums.EventType.UPDATED;

/**
 * Default implementation of {@link NewsService} to perform operations related to news resources
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultNewsService implements NewsService {

    private final NewsDmClient newsDmClient;
    private final NewsAuthorDetailProvider newsAuthorDetailProvider;
    private final NewsInstitutionalPageProvider newsInstitutionalPageProvider;
    private final ApplicationEventPublisher applicationEventPublisher;

    @NonNull
    @Override
    public OutputPageDto<OutputNewsDetailDto> findAll(@NonNull int pageNumber,
                                                      @NonNull int pageSize,
                                                      @NonNull String sort,
                                                      @NonNull Sort.Direction direction) {
        OutputPageDto<OutputNewsDto> outputPageNewsDto = newsDmClient.findAll(pageNumber, pageSize, sort, direction);
        if (outputPageNewsDto.getContent().isEmpty()) {
            return OutputPageDto.emptyWithPage(outputPageNewsDto.getPage());
        }
        return getDetailsForNews(outputPageNewsDto);
    }

    @NonNull
    private OutputPageDto<OutputNewsDetailDto> getDetailsForNews(@NotNull OutputPageDto<OutputNewsDto> detailProvider) {
        List<UUID> userIds = Stream.concat(
                detailProvider.getContent().stream()
                        .flatMap(dto -> Optional.ofNullable(dto.getRegisteredAuthorIds())
                                .orElse(Collections.emptyList())
                                .stream()
                                .map(UUID::fromString)),
                detailProvider.getContent().stream()
                        .map(OutputNewsDto::getCreatorId)
                        .filter(Objects::nonNull)).distinct().toList();

        List<UUID> institutionalPageIds = detailProvider.getContent().stream()
                .flatMap(dto -> Optional.ofNullable(dto.getInstitutionalPages()).map(institutionalPages ->
                                institutionalPages.stream().map(OutputInstitutionalPageDto::getId).toList())
                        .orElse(Collections.emptyList()).stream())
                .distinct().toList();

        Map<UUID, OutputNewsAuthorDto> userDetailByUserId = getUserDetailById(userIds);
        Map<UUID, OutputNewsInstitutionalPageDto> institutionalPageById = getInstitutionalPageById(institutionalPageIds);

        List<OutputNewsDetailDto> outputNewsDetailDtoList = new ArrayList<>();
        detailProvider.getContent().forEach(outputNewsDto -> {
            OutputNewsDetailDto outputNewsDetailDto = buildAndEnrichNewsDetailDto(outputNewsDto, userDetailByUserId, institutionalPageById);
            outputNewsDetailDtoList.add(outputNewsDetailDto);
        });

        return OutputPageDto
                .<OutputNewsDetailDto>builder()
                .content(outputNewsDetailDtoList)
                .page(detailProvider.getPage())
                .build();
    }

    @NonNull
    @Override
    public OutputNewsDetailDto findById(@NonNull UUID newsId) {
        OutputNewsDto outputNewsDto = newsDmClient.findById(newsId);
        return getDetailsForNews(outputNewsDto);
    }

    @NonNull
    @Override
    public OutputNewsDetailDto create(@NotNull InputCreateNewsDto inputCreateNewsDto) {
        OutputNewsDto outputNewsDto = newsDmClient.createNews(inputCreateNewsDto);
        OutputNewsDetailDto outputNewsDetailDto = getDetailsForNews(outputNewsDto);
        applicationEventPublisher.publishEvent(new NewsApplicationEvent(outputNewsDetailDto, CREATED));
        return outputNewsDetailDto;
    }

    @NonNull
    @Override
    public OutputNewsDetailDto updateById(@NotNull UUID newsId, @NotNull InputUpdateNewsDto inputUpdateNewsDto) {
        OutputNewsDto outputNewsDto = newsDmClient.updateNewsById(newsId, inputUpdateNewsDto);
        OutputNewsDetailDto outputNewsDetailDto = getDetailsForNews(outputNewsDto);
        applicationEventPublisher.publishEvent(new NewsApplicationEvent(outputNewsDetailDto, UPDATED));
        return outputNewsDetailDto;
    }

    @Override
    public void deleteById(@NotNull UUID newsId) {
        newsDmClient.deleteNewsById(newsId);
        OutputNewsDetailDto outputNewsDetailDto = OutputNewsDetailDto.builder().id(newsId).build();
        applicationEventPublisher.publishEvent(new NewsApplicationEvent(outputNewsDetailDto, DELETED));
    }

    @NonNull
    private OutputNewsDetailDto getDetailsForNews(@NotNull OutputNewsDto outputNewsDto) {
        List<UUID> userIds = Stream.concat(
                outputNewsDto.getRegisteredAuthorIds().stream().map(UUID::fromString),
                Stream.of(outputNewsDto.getCreatorId())
        ).distinct().toList();

        List<UUID> institutionalPageIds = Optional.ofNullable(outputNewsDto.getInstitutionalPages())
                .map(institutionalPages ->
                        institutionalPages.stream().map(OutputInstitutionalPageDto::getId).toList())
                .orElse(List.of());

        Map<UUID, OutputNewsAuthorDto> userDetailByUserId = getUserDetailById(userIds);
        Map<UUID, OutputNewsInstitutionalPageDto> institutionalPageById = getInstitutionalPageById(institutionalPageIds);

        return buildAndEnrichNewsDetailDto(outputNewsDto, userDetailByUserId, institutionalPageById);
    }

    @NonNull
    private Map<UUID, OutputNewsAuthorDto> getUserDetailById(@NotEmpty List<UUID> userIds) {
        try {
            OutputPageDto<OutputNewsAuthorDto> outputAuthorDetailPageDto = newsAuthorDetailProvider.getDetails(DetailRequest
                    .builder()
                    .resourceIds(userIds)
                    .pageRequestDto(PageRequestDto
                            .builder()
                            .pageNumber(0)
                            .pageSize(userIds.size())
                            .build())
                    .build());

            return outputAuthorDetailPageDto
                    .getContent()
                    .stream()
                    .collect(Collectors.toMap(OutputNewsAuthorDto::getId, Function.identity()));
        } catch (Exception e) {
            log.error("Failed to fetch user profiles for ids: {}", userIds, e);
            return userIds.stream()
                    .collect(Collectors.toMap(
                            id -> id,
                            id -> OutputNewsAuthorDto.builder().id(id).build()
                    ));
        }
    }

    @NonNull
    private Map<UUID, OutputNewsInstitutionalPageDto> getInstitutionalPageById(@NotEmpty List<UUID> institutionalPageIds) {
        try {
            OutputPageDto<OutputNewsInstitutionalPageDto> institutionalPageProviderDetails =
                    newsInstitutionalPageProvider.getDetails(DetailRequest
                            .builder()
                            .resourceIds(institutionalPageIds)
                            .pageRequestDto(PageRequestDto
                                    .builder()
                                    .pageNumber(0)
                                    .pageSize(institutionalPageIds.size())
                                    .build())
                            .build());

            return institutionalPageProviderDetails
                    .getContent()
                    .stream()
                    .collect(Collectors.toMap(OutputNewsInstitutionalPageDto::getId, Function.identity()));
        } catch (Exception e) {
            log.error("Failed to fetch institutional page details for ids: {}", institutionalPageIds, e);
            return institutionalPageIds.stream()
                    .collect(Collectors.toMap(
                            id -> id,
                            id -> OutputNewsInstitutionalPageDto.builder().id(id).build()
                    ));
        }
    }

    @NonNull
    private OutputNewsDetailDto buildAndEnrichNewsDetailDto(@NotNull OutputNewsDto outputNewsDto,
                                                            Map<UUID, OutputNewsAuthorDto> userDetailByUserId,
                                                            Map<UUID, OutputNewsInstitutionalPageDto> institutionalPageById) {
        OutputNewsDetailDto outputNewsDetailDto = buildNewsDetailDto(outputNewsDto);
        outputNewsDetailDto.setCreator(userDetailByUserId.get(outputNewsDto.getCreatorId()));

        List<OutputNewsAuthorDto> authors = getAuthors(outputNewsDto, userDetailByUserId);
        outputNewsDetailDto.setAuthors(authors);

        List<OutputNewsInstitutionalPageDto> institutionalPages = getInstitutionalPages(outputNewsDto, institutionalPageById);
        outputNewsDetailDto.setInstitutionalPages(institutionalPages);
        return outputNewsDetailDto;
    }

    private OutputNewsDetailDto buildNewsDetailDto(@NotNull OutputNewsDto newsDto) {
        return OutputNewsDetailDto.builder()
                .id(newsDto.getId())
                .title(newsDto.getTitle())
                .newsType(newsDto.getNewsType())
                .content(newsDto.getContent())
                .tags(newsDto.getTags())
                .image(newsDto.getImage())
                .creationTime(newsDto.getCreationTime())
                .updateTime(newsDto.getUpdateTime())
                .build();
    }

    private OutputNewsAuthorDto buildNewsAuthorDto(@NotNull OutputAuthorDto authorDto) {
        return OutputNewsAuthorDto.builder()
                .firstName(authorDto.getFirstName())
                .lastName(authorDto.getLastName())
                .orcid(authorDto.getOrcid())
                .email(authorDto.getEmail())
                .location(authorDto.getLocation())
                .build();
    }

    @NonNull
    private List<OutputNewsAuthorDto> getAuthors(@NotNull OutputNewsDto outputNewsDto,
                                                 Map<UUID, OutputNewsAuthorDto> userDetailByUserId) {
        List<OutputNewsAuthorDto> registeredAuthors = outputNewsDto.getRegisteredAuthorIds()
                .stream().map(s -> {
                    UUID authorId = UUID.fromString(s);
                    OutputNewsAuthorDto author = userDetailByUserId.get(authorId);
                    if (author == null) {
                        log.debug("News Author profile not found for ID: {}", authorId);
                        author = OutputNewsAuthorDto.builder().id(authorId).build();
                    }
                    return author;
                }).filter(Objects::nonNull).toList();

        return Stream.concat(
                outputNewsDto.getUnRegisteredAuthors().stream().map(this::buildNewsAuthorDto),
                registeredAuthors.stream()
        ).toList();
    }

    @NonNull
    private List<OutputNewsInstitutionalPageDto> getInstitutionalPages(@NotNull OutputNewsDto outputNewsDto,
                                                                       Map<UUID, OutputNewsInstitutionalPageDto> institutionalPageById) {
        return outputNewsDto.getInstitutionalPages()
                .stream().map(ipDto -> {
                    OutputNewsInstitutionalPageDto institutionalPage = institutionalPageById.get(ipDto.getId());
                    if (institutionalPage == null) {
                        log.debug("News Institutional Page not found for ID: {}", ipDto.getId());
                        institutionalPage = OutputNewsInstitutionalPageDto.builder().id(ipDto.getId()).name(ipDto.getName()).build();
                    }
                    return institutionalPage;
                }).filter(Objects::nonNull).toList();
    }
}
