package com.finconsgroup.itserr.marketplace.news.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import com.finconsgroup.itserr.marketplace.news.dm.entity.converter.StringListConverter;
import com.finconsgroup.itserr.marketplace.news.dm.model.InstitutionalPageModel;
import jakarta.persistence.CascadeType;
import jakarta.persistence.Column;
import jakarta.persistence.Convert;
import jakarta.persistence.Entity;
import jakarta.persistence.Lob;
import jakarta.persistence.OneToMany;
import jakarta.persistence.Table;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;
import org.hibernate.annotations.JdbcTypeCode;
import org.hibernate.type.SqlTypes;

import java.time.Instant;
import java.util.List;
import java.util.UUID;

import static com.finconsgroup.itserr.marketplace.news.dm.entity.NewsEntity.CONTENT_LENGTH;
import static com.finconsgroup.itserr.marketplace.news.dm.entity.NewsEntity.TITLE_LENGTH;
import static com.finconsgroup.itserr.marketplace.news.dm.entity.NewsEntity.TYPE_LENGTH;

/**
 * This entity represents a soft deleted {@link NewsEntity}.
 * But it doesn't have unique constraints on name field.
 */
@Entity
@Table(name = "archived_news")
@Data
@SuperBuilder
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class ArchivedNewsEntity extends AbstractUUIDEntity {

    /**
     * The title of the news.
     */
    @Column(name = "title", nullable = false, length = TITLE_LENGTH)
    private String title;

    /**
     * The ID of the user who created the news.
     */
    @Column(name = "creator_id", nullable = false)
    private UUID creatorId;

    /**
     * The type of news.
     */
    @Column(name = "news_type", length = TYPE_LENGTH)
    private String newsType;

    /**
     * The content of the news.
     */
    @Column(name = "content", length = CONTENT_LENGTH)
    private String content;

    /**
     * List of tags associated with the news.
     */
    @Lob
    @Convert(converter = StringListConverter.class)
    @Column(name = "tags")
    private List<String> tags;

    /**
     * The JSON structure for Institutional Page.
     */
    @JdbcTypeCode(SqlTypes.JSON)
    @Column(name = "institutional_pages", columnDefinition = "json")
    private List<InstitutionalPageModel> institutionalPages;

    /**
     * The document ID of the image.
     */
    @Column(name = "image_id")
    private UUID imageId;

    /**
     * Public URL of the image.
     */
    @Column(name = "image_url")
    private String imageUrl;

    /**
     * List of registered author IDs.
     */
    @Lob
    @Convert(converter = StringListConverter.class)
    @Column(name = "registered_author_ids")
    private List<String> registeredAuthorIds;

    /**
     * List of unregistered author.
     */
    @OneToMany(mappedBy = "archivedNews", cascade = CascadeType.ALL, orphanRemoval = true)
    private List<ArchivedAuthorEntity> unRegisteredAuthors;

    /**
     * The timestamp when the news was created.
     */
    @Column(name = "creation_time", nullable = false, updatable = false)
    private Instant creationTime;

    /**
     * The timestamp when the news was last updated.
     */
    @Column(name = "update_time", nullable = false)
    private Instant updateTime;

    @Column(name = "version", nullable = false)
    private long version;

    @Override
    public void prePersist() {
        super.prePersist();
        Instant now = Instant.now();
        if (creationTime == null) {
            creationTime = now;
        }
        if (updateTime == null) {
            updateTime = now;
        }
    }

}
