package com.finconsgroup.itserr.marketplace.news.dm.service;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.news.dm.dto.InputCreateNewsDto;
import com.finconsgroup.itserr.marketplace.news.dm.dto.InputUpdateNewsDto;
import com.finconsgroup.itserr.marketplace.news.dm.dto.OutputNewsDto;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;

import java.util.UUID;

/**
 * Service for handling business logic related to News. This interface provides methods to retrieve, create, update, and delete News,
 *
 * <p>Example usage:
 * <pre>
 * Page<OutputNewsDto> News = NewsService.findAll(pageable);
 * </pre>
 * </p>
 */
public interface NewsService {

    /**
     * Retrieves a paginated list of all News.
     *
     * @param pageable the pagination information
     * @return a page of OutputNewsDto representing the News
     */
    @NonNull
    Page<OutputNewsDto> findAll(@NonNull Pageable pageable);

    /**
     * Retrieves news by id.
     *
     * @param newsId the id of the news
     * @return an OutputNewsDto representing the news
     * @throws WP2ResourceNotFoundException if no news is found with the given id
     */
    @NonNull
    OutputNewsDto findById(@NonNull UUID newsId);

    /**
     * Creates new news for user using the provided input data.
     *
     * @param userId             the id of the user
     * @param inputCreateNewsDto the input DTO containing news details
     */
    @NonNull
    OutputNewsDto create(@NonNull UUID userId, @NonNull InputCreateNewsDto inputCreateNewsDto);

    /**
     * Updates existing news by id with the provided data.
     *
     * @param userId             the id of the user
     * @param newsId             the id of the news to update
     * @param inputUpdateNewsDto the input DTO containing updated  news details
     * @return an OutputNewsDto newsDto representing the updated news
     * @throws WP2ResourceNotFoundException if no news is found with the given id
     */
    @NonNull
    OutputNewsDto updateById(@NonNull UUID userId,
                             @NonNull UUID newsId,
                             @NonNull InputUpdateNewsDto inputUpdateNewsDto);


    /**
     * Deletes news by id. It will move the news entity on a separated table containing deleted records.
     *
     * @param userId the id of the user
     * @param newsId the id of the news to delete
     * @throws WP2ResourceNotFoundException if no news is found with the given id
     */
    @NonNull
    void deleteById(@NonNull UUID userId, @NonNull UUID newsId);

}
