package com.finconsgroup.itserr.marketplace.notification.bs.client;

import com.finconsgroup.itserr.marketplace.notification.bs.client.dto.InputCreateUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.bs.client.dto.InputPatchUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.bs.client.dto.InputPatchUserNotificationsDto;
import com.finconsgroup.itserr.marketplace.notification.bs.client.dto.OutputLocalizedUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.bs.client.dto.OutputNotificationEmailTemplateDto;
import com.finconsgroup.itserr.marketplace.notification.bs.client.dto.OutputNotificationTemplateDto;
import com.finconsgroup.itserr.marketplace.notification.bs.client.dto.OutputPatchUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.bs.client.dto.OutputUserNotificationDto;
import jakarta.validation.Valid;
import org.springframework.http.MediaType;
import org.springframework.lang.NonNull;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;

import java.util.List;
import java.util.UUID;

/**
 * Notification DM microservice client.
 */
public interface NotificationDmClient {

    /**
     * Retrieves a user notification by id.
     *
     * @param id             the id of the notification to retrieve
     * @param acceptLanguage Accept-Language header value.
     * @return the found {@link OutputUserNotificationDto} and HTTP status 200 (OK)
     */
    @GetMapping(value = "/notification/notifications/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputLocalizedUserNotificationDto findById(
            @PathVariable("id") UUID id,
            @RequestHeader(name = "Accept-Language", required = false) String acceptLanguage);

    /**
     * Creates a new user notification.
     *
     * @param inputCreateUserNotificationDto the input data transfer object containing user notification details
     * @param acceptLanguage                 Accept-Language header value.
     * @return the created notification
     */
    @PostMapping(
            value = "/notification/notifications",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    List<OutputUserNotificationDto> create(
            @Valid @RequestBody InputCreateUserNotificationDto inputCreateUserNotificationDto,
            @RequestHeader(name = "Accept-Language", required = false) String acceptLanguage);

    /**
     * Patches all existing user notifications.
     *
     * @param request the patching request
     */
    @PatchMapping(
            value = "/notification/notifications",
            consumes = MediaType.APPLICATION_JSON_VALUE
    )
    void patchAll(
            @Valid @RequestBody InputPatchUserNotificationsDto request
    );

    /**
     * Patches an existing user notification by id.
     *
     * @param id                            the id of the notification to patch
     * @param inputPatchUserNotificationDto the input data transfer object containing patched notification details
     * @return the patched {@link OutputPatchUserNotificationDto} and HTTP status 200 (OK)
     */
    @PatchMapping(
            value = "/notification/notifications/{id}",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    OutputPatchUserNotificationDto patchById(
            @NonNull @PathVariable("id") UUID id,
            @Valid @RequestBody InputPatchUserNotificationDto inputPatchUserNotificationDto
    );

    /**
     * Deletes a user notification by id.
     *
     * @param id the id of the notification to delete
     */
    @DeleteMapping(value = "/notification/notifications/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    void deleteById(@NonNull @PathVariable("id") UUID id);

    /**
     * Get the templates of a given notification.
     *
     * @param notificationType the notificationType of the templates we need to get
     */
    @GetMapping(value = "/notification/notifications/templates/{notificationType}", produces = MediaType.APPLICATION_JSON_VALUE)
    List<OutputNotificationTemplateDto> getMessageTemplatesByType(@NonNull @PathVariable("notificationType") String notificationType);

    /**
     * Retrieves all notification email templates.
     *
     * @return the found {@link List<OutputNotificationEmailTemplateDto> } and HTTP status 200 (OK)
     */
    @GetMapping(value = "/notification/notifications/templates/email", produces = MediaType.APPLICATION_JSON_VALUE)
    List<OutputNotificationEmailTemplateDto> getEmailTemplates();
}
