package com.finconsgroup.itserr.marketplace.notification.dm.controller;

import com.finconsgroup.itserr.marketplace.notification.dm.api.UserNotificationApi;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.InputCreateUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.InputPatchUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.InputPatchUserNotificationsDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.OutputLocalizedUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.OutputNotificationEmailTemplateDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.OutputNotificationTemplateDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.OutputPatchUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.OutputUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.UserDto;
import com.finconsgroup.itserr.marketplace.notification.dm.service.LocalizedUserNotificationService;
import com.finconsgroup.itserr.marketplace.notification.dm.service.NotificationTemplateService;
import com.finconsgroup.itserr.marketplace.notification.dm.service.UserNotificationService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.UUID;

/**
 * REST controller class for handling user Notification related API requests.
 *
 * <p>Implements the {@link UserNotificationApi} interface.</p>
 */
@Slf4j
@RequiredArgsConstructor
@RestController
public class UserNotificationController implements UserNotificationApi {

    private final UserNotificationService userNotificationService;
    private final LocalizedUserNotificationService localizedUserNotificationService;
    private final NotificationTemplateService notificationTemplateService;

    @Override
    public Page<OutputLocalizedUserNotificationDto> findAll(
            final Boolean read,
            final Boolean archived,
            final int pageNumber,
            final int pageSize,
            final String acceptLanguage
    ) {
        final UserDto user = UserDto.fromSecurityOrThrow();
        Pageable pageable = PageRequest.of(pageNumber, pageSize);
        return localizedUserNotificationService.findAll(
                user,
                read,
                archived,
                pageable,
                acceptLanguage);
    }

    @Override
    public void patchAll(final InputPatchUserNotificationsDto request) {
        final UserDto user = UserDto.fromSecurityOrThrow();
        userNotificationService.patchAll(
                user,
                request);
    }

    @Override
    public OutputLocalizedUserNotificationDto findById(
            final UUID notificationId,
            final String acceptLanguage) {
        final UserDto user = UserDto.fromSecurityOrThrow();
        return localizedUserNotificationService.findByUser(
                user,
                notificationId,
                acceptLanguage);
    }

    @Override
    public List<OutputUserNotificationDto> createNotification(
            final InputCreateUserNotificationDto inputCreateUserNotificationDto) {
        return userNotificationService.create(inputCreateUserNotificationDto);
    }

    @Override
    public OutputPatchUserNotificationDto patchUserNotificationById(
            final UUID notificationId,
            final InputPatchUserNotificationDto inputPatchUserNotificationDto
    ) {
        final UserDto user = UserDto.fromSecurityOrThrow();
        return userNotificationService.patchById(
                user,
                notificationId,
                inputPatchUserNotificationDto);
    }

    @Override
    public void deleteUserNotificationById(final UUID notificationId) {
        final UserDto user = UserDto.fromSecurityOrThrow();
        userNotificationService.deleteById(
                user,
                notificationId);
    }

    @Override
    public List<OutputNotificationTemplateDto> getTemplatesByType(String notificationType) {
        return notificationTemplateService.findByType(notificationType);
    }

    @Override
    public List<OutputNotificationEmailTemplateDto> getEmailTemplates() {
        return notificationTemplateService.findAllEmailTemplates();
    }

}
