package com.finconsgroup.itserr.marketplace.notification.dm.repository;

import com.finconsgroup.itserr.marketplace.notification.dm.entity.UserNotificationEntity;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.util.Optional;
import java.util.UUID;

/**
 * {@link UserNotificationRepository} repository.
 */
@Repository
public interface UserNotificationRepository extends JpaRepository<UserNotificationEntity, UUID>, JpaSpecificationExecutor<UserNotificationEntity> {

    /**
     * Retrieves a specific user notification entity based on the provided user ID and notification ID.
     *
     * @param userId the id of the user to whom the notification belongs
     * @param userUsername the username of the user to whom the notification belongs
     * @param userEmail the email of the user to whom the notification belongs
     * @param notificationId the unique identifier of the notification to be retrieved
     * @return an {@link Optional} containing the matching {@link UserNotificationEntity} if found, or an empty {@link Optional} if no match exists
     */
    @Query("""
            SELECT n
            FROM UserNotificationEntity n
            WHERE n.id = :id
              AND LOWER(n.user) IN (LOWER(:userId), LOWER(:userUsername), LOWER(:userEmail))
            """)
    Optional<UserNotificationEntity> findByUserAndId(
            @Param("userId") String userId,
            @Param("userUsername") String userUsername,
            @Param("userEmail") String userEmail,
            @Param("id") UUID notificationId);

    /**
     * Updates the 'read' status of all notification entities for a specific user.
     *
     * @param userId the unique identifier of the user for whom the notifications will be updated
     * @param userUsername the username of the user to whom the notification belongs
     * @param userEmail the email of the user to whom the notification belongs
     * @param read the new read status to be applied to the notifications (true for read, false for unread)
     */
    @Modifying
    @Query("""
            UPDATE UserNotificationEntity n
            SET n.read = :read
            WHERE LOWER(user) = LOWER(:userId)
               OR LOWER(user) = LOWER(:userUsername)
               OR LOWER(user) = LOWER(:userEmail)
            """)
    void updateRead(
            @Param("userId") String userId,
            @Param("userUsername") String userUsername,
            @Param("userEmail") String userEmail,
            @Param("read") boolean read);

}
