package com.finconsgroup.itserr.marketplace.notificationfeeder.bs.event;

import com.finconsgroup.itserr.marketplace.notificationfeeder.bs.event.filter.ConsumerHandlerFilterRegistry;
import com.finconsgroup.itserr.messaging.consumer.handler.ConsumerMessageHandler;
import io.cloudevents.CloudEvent;
import io.cloudevents.CloudEventData;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.util.HashSet;
import java.util.Set;

/**
 * Handles message multiplexing for a specific stream by distributing incoming CloudEvents to registered handlers. This class implements the
 * {@link ConsumerMessageHandler} interface to process messages from a specific stream and forwards them to all registered
 * {@link JsonNotificationConsumerMessageHandler}.
 */
@Slf4j
@RequiredArgsConstructor
public class StreamMultiplexerConsumerMessageHandler implements ConsumerMessageHandler {

    /** The name of the stream this handler is responsible for */
    @Getter
    private final String stream;

    /** Set of handlers that will process the multiplexed messages */
    private final Set<NotificationConsumerMessageHandler> handlers = new HashSet<>();

    /** Registry for managing and applying filters on consumer handlers when processing messages. */
    private final ConsumerHandlerFilterRegistry handlersFilterRegistry;

    /**
     * Adds a new handler to process messages from this stream. The operation is synchronized to ensure thread safety.
     *
     * @param handler the notification message handler to add
     */
    public void addHandler(final JsonNotificationConsumerMessageHandler handler) {
        synchronized (handlers) {
            handlers.add(handler);
        }
    }

    /**
     * Processes an incoming CloudEvent by parsing its JSON data and distributing it to all registered handlers. If the event contains no data or cannot be
     * parsed, the method will log appropriate messages and return.
     *
     * @param cloudEvent the CloudEvent to process
     */
    @Override
    public void handleMessage(final CloudEvent cloudEvent) {

        // Check event data
        final CloudEventData rawData = cloudEvent.getData();
        if (rawData == null) {
            return;
        }

        // Log
        if (log.isTraceEnabled()) {
            final String data = new String(rawData.toBytes());
            log.trace("Received event: id={}, stream={}, type={}, message={}", cloudEvent.getId(), stream, cloudEvent.getType(), data);
        } else {
            log.debug("Received event: id={}, stream={}, type={}", cloudEvent.getId(), stream, cloudEvent.getType());
        }

        // Multiplex message to all handlers
        for (final NotificationConsumerMessageHandler handler : handlers) {
            try {

                // Init
                final String handlerName = handler.getName();

                // Check that handler supports the message
                if (handlersFilterRegistry.supports(handler, cloudEvent)) {

                    // Handle message
                    log.trace("Propagating to handler {}", handlerName);
                    handler.handleMessage(cloudEvent);

                } else {

                    log.trace("Skipping handler {} (does not support event)", handlerName);

                }

            } catch (Exception e) {
                log.error("Error handling event: stream={}", stream, e);
            }
        }

    }

}
