package com.finconsgroup.itserr.marketplace.search.bs.client.dm;

import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;

/**
 * Client for communication with the Search DM service,
 * managing catalog and user profile indexing operations.
 */
public interface SearchDmClient {

    /**
     * Creates or updates a catalog document in the search index.
     *
     * @param id  the identifier of the catalog document
     * @param dto the payload representing the catalog entry
     * @throws feign.FeignException.InternalServerError if an internal error occurs
     */
    @PutMapping("/catalog/{id}")
    void upsertCatalog(@PathVariable("id") String id, @RequestBody Object dto);

    /**
     * Deletes a catalog document from the search index.
     *
     * @param id the identifier of the catalog document to delete
     * @throws feign.FeignException.InternalServerError if an internal error occurs
     */
    @DeleteMapping("/catalog/{id}")
    void deleteCatalog(@PathVariable("id") String id);

    /**
     * Creates or updates a user profile document in the search index.
     *
     * @param id  the identifier of the user profile
     * @param dto the payload representing the user profile entry
     * @throws feign.FeignException.InternalServerError if an internal error occurs
     */
    @PutMapping("/profile/{id}")
    void upsertUserProfile(@PathVariable("id") String id, @RequestBody Object dto);

    /**
     * Deletes a user profile document from the search index.
     *
     * @param id the identifier of the user profile to delete
     * @throws feign.FeignException.InternalServerError if an internal error occurs
     */
    @DeleteMapping("/profile/{id}")
    void deleteUserProfile(@PathVariable("id") String id);

    /**
     * Creates or updates an institutional page document in the search index.
     *
     * @param id  the identifier of the institutional page
     * @param dto the payload representing the institutional page entry
     * @throws feign.FeignException.InternalServerError if an internal error occurs
     */
    @PutMapping("/institutional-page/{id}")
    void upsertInstitutionalPage(@PathVariable("id") String id, @RequestBody Object dto);

    /**
     * Deletes an institutional page document from the search index.
     *
     * @param id the identifier of the institutional page to delete
     * @throws feign.FeignException.InternalServerError if an internal error occurs
     */
    @DeleteMapping("/institutional-page/{id}")
    void deleteInstitutionalPage(@PathVariable("id") String id);

    /**
     * Creates or updates an event document in the search index.
     *
     * @param id  the identifier of the event
     * @param dto the payload representing the event entry
     * @throws feign.FeignException.InternalServerError if an internal error occurs
     */
    @PutMapping("/event/{id}")
    void upsertEvent(@PathVariable("id") String id, @RequestBody Object dto);

    /**
     * Deletes an event document from the search index.
     *
     * @param id the identifier of the event to delete
     * @throws feign.FeignException.InternalServerError if an internal error occurs
     */
    @DeleteMapping("/event/{id}")
    void deleteEvent(@PathVariable("id") String id);

    /**
     * Creates or updates a news document in the search index.
     *
     * @param id  the identifier of the event
     * @param dto the payload representing the news entry
     * @throws feign.FeignException.InternalServerError if an internal error occurs
     */
    @PutMapping("/news/{id}")
    void upsertNews(@PathVariable("id") String id, @RequestBody Object dto);

    /**
     * Deletes a news document from the search index.
     *
     * @param id the identifier of the news to delete
     * @throws feign.FeignException.InternalServerError if an internal error occurs
     */
    @DeleteMapping("/news/{id}")
    void deleteNews(@PathVariable("id") String id);

    /**
     * Creates or updates a discussion document in the search index.
     *
     * @param id  the identifier of the discussion
     * @param dto the payload representing the discussion entry
     * @throws feign.FeignException.InternalServerError if an internal error occurs
     */
    @PutMapping("/discussion/{id}")
    void upsertDiscussion(@PathVariable("id") String id, @RequestBody Object dto);

    /**
     * Deletes a discussion from the search index.
     *
     * @param id the identifier of the discussion to delete
     * @throws feign.FeignException.InternalServerError if an internal error occurs
     */
    @DeleteMapping("/discussion/{id}")
    void deleteDiscussion(@PathVariable("id") String id);

}
