package com.finconsgroup.itserr.marketplace.search.dm.converter;

import com.finconsgroup.itserr.marketplace.search.dm.constant.DateTimeFormats;
import org.springframework.data.elasticsearch.core.mapping.PropertyValueConverter;
import org.springframework.lang.NonNull;

import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;

/**
 * Converter class for converting between {@link Instant} and {@link String} for any
 * search document.
 * As these are not auto registered, will need to use
 * {@link org.springframework.data.elasticsearch.annotations.ValueConverter} annotations to apply
 * it to the specific fields of the document.
 */
public class InstantPropertyValueConverter implements PropertyValueConverter {

    /**
     * Formatter to always convert instants in the format supported by Open Search
     */
    private static final DateTimeFormatter FORMATTER = DateTimeFormatter
            .ofPattern(DateTimeFormats.INSTANT_OPEN_SEARCH)
            .withZone(ZoneId.of("UTC"));

    @NonNull
    @Override
    public Object read(@NonNull Object value) {
        if (value instanceof String strValue) {
            return Instant.parse(strValue);
        }
        return Instant.parse(value.toString());
    }

    @NonNull
    @Override
    public Object write(@NonNull Object value) {
        if (value instanceof Instant instant) {
            return FORMATTER.format(instant);
        }

        return value.toString();
    }
}
