package com.finconsgroup.itserr.marketplace.search.dm.service;

import com.finconsgroup.itserr.marketplace.core.web.bean.FilterProperties;
import com.finconsgroup.itserr.marketplace.core.web.bean.QueryFilter;
import com.finconsgroup.itserr.marketplace.core.web.enums.FilterOperator;
import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.core.web.utils.FilterUtils;
import com.finconsgroup.itserr.marketplace.search.dm.bean.ContainsAnyTermsRequest;
import com.finconsgroup.itserr.marketplace.search.dm.bean.ContainsAnyTermsWithFiltersRequest;
import com.finconsgroup.itserr.marketplace.search.dm.bean.PostProcessFilterResult;
import com.finconsgroup.itserr.marketplace.search.dm.bean.QueryRequest;
import com.finconsgroup.itserr.marketplace.search.dm.bean.QuerySearchFields;
import com.finconsgroup.itserr.marketplace.search.dm.bean.SearchRequest;
import com.finconsgroup.itserr.marketplace.search.dm.bean.TopHitsAggregationRequest;
import com.finconsgroup.itserr.marketplace.search.dm.config.SearchProperties;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchAutoCompleteDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchDto;
import com.finconsgroup.itserr.marketplace.search.dm.repository.CustomAggregationRepository;
import com.finconsgroup.itserr.marketplace.search.dm.repository.CustomQueryRepository;
import com.finconsgroup.itserr.marketplace.search.dm.util.SortUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.lang.NonNull;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * Service for search related operations.
 *
 * @param <T> the type of the document
 * @param <L> type of local search result
 */
public interface SearchService<T, L> {

    /**
     * Format to combine sort field name and direction.
     * The arguments to be provided are sort field name and sort direction.
     */
    String SORT_FIELD_FORMAT = "%s:%s";
    /**
     * Format to derive the keyword inner field for text fields so that it can be used for sorting/filtering.
     * The argument to be provided is the text field name.
     */
    String TEXT_INNER_KEYWORD_FIELD_FORMAT = "%s.keyword";
    /**
     * Filter key to request only documents created by the logged-in user.
     * The value should be boolean i.e. true or false.
     */
    String FILTER_KEY_MINE_ONLY = "mineOnly";

    /**
     * Performs search for the document index for {@code T} based on the terms and filters, and
     * returns the results as page of dto {@code R}
     *
     * @param searchRequest the search request to search for
     * @param sourceFields  the source fields to fetch
     * @param mapper        the mapper to map the document to result type
     * @param pageable      the page to return
     * @return the page of {@code R} of matching documents
     */
    @NonNull
    default <R> Page<R> search(@NonNull SearchRequest searchRequest,
                               List<String> sourceFields,
                               Function<T, R> mapper,
                               @NonNull Pageable pageable) {
        QueryRequest<T> queryRequest = buildQueryRequest(searchRequest, sourceFields);
        return getCustomQueryRepository().findPageForQuery(queryRequest, pageable).map(mapper);
    }

    /**
     * Search for documents for the provided terms string which can contain multiple terms.
     * It additionally aggregates the results by the aggregation request's term.
     * It returns the results aggregated by term and limited to the page size.
     *
     * @param searchRequest the search request to search for
     * @param sourceFields  the source fields to fetch
     * @param topHitsLimit  the limit on number of records to return for each term
     * @param aggregation   the aggregation properties to apply
     * @param mapper        the mapper to map the document to result type
     * @return the page of {@code R} of matching documents
     */
    @NonNull
    default <R> Map<String, List<R>> searchByAggregation(
        @NonNull SearchRequest searchRequest,
        @NonNull List<String> sourceFields,
        int topHitsLimit,
        @NonNull SearchProperties.AggregationProperties aggregation,
        @NonNull Function<T, R> mapper) {

        SearchProperties searchProperties = getSearchProperties();
        QueryRequest<T> queryRequest = buildQueryRequest(searchRequest, sourceFields);
        TopHitsAggregationRequest<T> topHitsAggregationRequest = TopHitsAggregationRequest
            .<T>builder()
            .name(aggregation.name())
            .term(aggregation.term())
            .topHitsLimit(topHitsLimit)
            .sourceFields(sourceFields)
            .includeScore(searchProperties.includeScore())
            .documentClass(getDocumentClass())
            .build();
        Map<String, List<T>> topHitsByAggregationTerm = getCustomAggregationRepository().findTopHitsForQueryByTerm(
            queryRequest, topHitsAggregationRequest);

        Map<String, List<R>> topHitResultsByAggregationTerm = new LinkedHashMap<>();
        topHitsByAggregationTerm.forEach((aggregationTerm, topHits) ->
            topHitResultsByAggregationTerm.put(aggregationTerm, topHits.stream().map(mapper).collect(Collectors.toList()))
        );
        return topHitResultsByAggregationTerm;
    }

    default QueryRequest<T> buildQueryRequest(SearchRequest searchRequest, List<String> sourceFields) {
        SearchProperties searchProperties = getSearchProperties();
        ContainsAnyTermsRequest<T> termsQueryRequest = null;
        if (searchRequest.terms() != null && !searchRequest.terms().isEmpty()) {
            termsQueryRequest = ContainsAnyTermsRequest
                .<T>builder()
                .documentClass(getDocumentClass())
                .terms(searchRequest.terms())
                .sourceFields(sourceFields)
                .nestedPaths(searchProperties.nestedPaths())
                .searchFields(Optional.ofNullable(searchRequest.termSearchFields()).orElse(searchProperties.termSearchFields()))
                .useWildcard(searchProperties.useWildcard())
                .includeScore(searchProperties.includeScore())
                .build();
        }
        return ContainsAnyTermsWithFiltersRequest
            .<T>builder()
            .documentClass(getDocumentClass())
            .sourceFields(sourceFields)
            .nestedPaths(searchProperties.nestedPaths())
            .ids(searchRequest.ids())
            .containsAnyTermsRequest(termsQueryRequest)
            .queryFilters(searchRequest.queryFilters())
            .includeScore(searchProperties.includeScore())
            .build();
    }

    /**
     * Maps the text field name to corresponding keyword field name to be used for sorting.
     *
     * @param searchProperties the search properties
     * @return the map containing text field and corresponding keyword field name
     */
    @NonNull
    default Map<String, String> buildSortFilterPropertyMap(@NonNull SearchProperties searchProperties, @NonNull Map<String, String> defaultSortPropertyMap) {
        Map<String, String> sortFilterPropertyMap = new HashMap<>();
        if (searchProperties.sortFilterPropertyMap() != null) {
            sortFilterPropertyMap.putAll(searchProperties.sortFilterPropertyMap());
        }
        searchProperties.termSearchFields().stream()
                        .filter(qsf -> QuerySearchFields.TYPE_TEXT.equals(qsf.fieldType()))
                        .findFirst()
                        .map(QuerySearchFields::fieldNames)
                        .ifPresent(textFields ->
                            textFields.forEach(field ->
                                sortFilterPropertyMap.put(field, TEXT_INNER_KEYWORD_FIELD_FORMAT.formatted(field))));
        // some default sort field mappings for internal fields
        sortFilterPropertyMap.putAll(defaultSortPropertyMap);
        return sortFilterPropertyMap;
    }

    /**
     * Builds a new pageable instance by applying the sort, mapping the field names, if required.
     * Also, if there is no sort provided, then applies the default sort, as per search configuration.
     *
     * @param pageable the input pageable with sort information
     * @param searchProperties the search properties
     * @param sortPropertyMap the sort property map, to rename fields for sorting
     * @return the pageable instance
     */
    @NonNull
    default Pageable applySort(@NonNull Pageable pageable,
                               @NonNull SearchProperties searchProperties,
                               @NonNull Map<String, String> sortPropertyMap) {
        List<String> sortFields = pageable.getSort().stream().map(
            order -> SORT_FIELD_FORMAT.formatted(order.getProperty(), order.getDirection())
        ).toList();
        if (sortFields.isEmpty()) {
            sortFields = searchProperties.defaultSortFields();
        }
        Sort sort = SortUtils.buildSort(sortFields, searchProperties.defaultSortFieldSeparator(),
            sortPropertyMap);
        return PageRequest.of(pageable.getPageNumber(), pageable.getPageSize(), sort);
    }

    /**
     * Builds a list of query filters based on the input filters and field name map
     *
     * @param filters the filter string to map
     * @param filterPropertyMap the property map, to rename fields for filtering
     * @return the list of {@link QueryFilter}
     */
    @NonNull
    default List<QueryFilter> buildQueryFilters(String filters, @NonNull Map<String, String> filterPropertyMap) {
        Optional<FilterProperties> optFilterProperties = Optional.ofNullable(getSearchProperties().filter());
        List<QueryFilter> queryFilters = FilterUtils.buildQueryFilters(filters,
            optFilterProperties.map(FilterProperties::separator).orElse(null),
            optFilterProperties.map(FilterProperties::keyValueSeparator).orElse(null),
            optFilterProperties.map(FilterProperties::valueSeparator).orElse(null),
            filterPropertyMap
        );
        return postProcessQueryFilters(queryFilters);
    }

    /**
     * Applies some common post-processing for input filters.
     *
     * @param queryFilters the query filters to process
     * @return the processed query filters list.
     */
    @NonNull
    default List<QueryFilter> postProcessQueryFilters(@NonNull List<QueryFilter> queryFilters) {
        List<QueryFilter> processedFilters = new ArrayList<>(queryFilters.size());
        for (QueryFilter queryFilter : queryFilters) {
            PostProcessFilterResult postProcessFilterResult = postProcessQueryFilter(queryFilter);
            if (postProcessFilterResult.processed()) {
                if (!postProcessFilterResult.skip()) {
                    // if the filter should not be skipped, then
                    // add either the post processed filter if available or the input filter itself
                    processedFilters.add(Optional.ofNullable(postProcessFilterResult.queryFilter()).orElse(queryFilter));
                }
            } else {
                // if the filter was not processed, then add input filter itself
                processedFilters.add(queryFilter);
            }
        }
        return processedFilters;
    }

    /**
     * Applies some common post-processing for input filter.
     * e.g. filter to return documents created by currently logged-in user only
     *
     * @param queryFilter the query filter to process
     * @return the processed query filter, {@link Optional#empty()} otherwise.
     */
    @NonNull
    default PostProcessFilterResult postProcessQueryFilter(@NonNull QueryFilter queryFilter) {
        PostProcessFilterResult.PostProcessFilterResultBuilder resultBuilder = PostProcessFilterResult
                .builder();
        if (FILTER_KEY_MINE_ONLY.equals(queryFilter.fieldName())) {
            postProcessMineOnlyQueryFilter(queryFilter, resultBuilder);
        }

        return resultBuilder.build();
    }

    /**
     * Applies the mineOnly filter to return documents created by currently logged-in user only.
     * The implementation classes can override this behavior if needed.
     *
     * @param queryFilter the query filter to process
     * @param resultBuilder the result builder that can be used to return appropriate result
     */
    default void postProcessMineOnlyQueryFilter(@NonNull QueryFilter queryFilter,
                                                @NonNull PostProcessFilterResult.PostProcessFilterResultBuilder resultBuilder) {
        resultBuilder.processed(true);

        // only process the filter if mineOnly=true
        // ignoring all other values and operators for this filter key
        if (FilterOperator.EQ == queryFilter.operator()
                && queryFilter.filterValues().size() == 1
                && Boolean.parseBoolean(queryFilter.filterValues().getFirst())) {
            Optional<UUID> userIdOpt = JwtTokenHolder.getUserId();
            Optional<String> creatorUserIdFieldName = getCreatorUserIdFieldName();
            // only apply the creator user id filter if
            // 1. User is logged-in
            // 2. Document has field to represent user who created it
            if (userIdOpt.isPresent() && creatorUserIdFieldName.isPresent()) {
                QueryFilter processedQueryFilter = QueryFilter.builder()
                        .fieldName(creatorUserIdFieldName.get())
                        .operator(FilterOperator.EQ)
                        .filterValues(List.of(userIdOpt.get().toString()))
                        .build();
                resultBuilder.skip(false).queryFilter(processedQueryFilter);
            }
        }
    }

    /**
     * Search for documents for the provided title string which can contain multiple terms.
     * It returns the results in suitable format that can be used for autocompletion.
     *
     * @param terms the terms to search for
     * @return List of {@link OutputGlobalSearchAutoCompleteDto} of matching documents
     */
    @NonNull
    List<OutputGlobalSearchAutoCompleteDto> getAutoCompletions(@NonNull String terms);

    /**
     * Performs local search for the document index for {@code T} based on the terms and filters, and
     * returns the results as page of dto {@code R}
     *
     * @param terms    the terms to search for
     * @param filters  the filters to apply
     * @param pageable the page to return
     * @return the page of {@code R} of matching documents
     */
    @NonNull
    Page<L> getLocalSearch(String terms, String filters, @NonNull Pageable pageable);

    /**
     * Search for documents for the provided title string which can contain multiple terms.
     *
     * @param terms the terms to search for
     * @return List of {@link OutputGlobalSearchDto} of matching documents
     */
    @NonNull
    List<OutputGlobalSearchDto> getSearch(@NonNull String terms);

    /**
     * Returns the class representing the type of the documents to search for
     *
     * @return Class of document type
     */
    @NonNull
    Class<T> getDocumentClass();

    /**
     * Returns the search properties for related configuration
     *
     * @return SearchProperties
     */
    @NonNull
    SearchProperties getSearchProperties();

    /**
     * Returns the custom query repository for the document
     *
     * @return CustomQueryRepository
     */
    @NonNull
    CustomQueryRepository getCustomQueryRepository();

    /**
     * Returns the custom aggregation repository for the document
     *
     * @return CustomAggregationRepository
     */
    @NonNull
    CustomAggregationRepository getCustomAggregationRepository();

    /**
     * Returns the field name representing the creator/maintainer of the document.
     * The default implementation returns {@link Optional#empty()}.
     * The specific implementation class must override and return relevant field, if applicable.
     *
     * @return {@link Optional}
     */
    default Optional<String> getCreatorUserIdFieldName() {
        return Optional.empty();
    }

}
