package com.finconsgroup.itserr.marketplace.userprofile.bs.client;

import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputCreateDocumentDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.OutputCreatedDocumentDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.OutputDocumentPublicLinkDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.OutputUpdatedDocumentDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.FolderDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.InputFolderCreateDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.WorkspaceDto;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import jakarta.validation.Valid;
import jakarta.validation.constraints.NotBlank;
import org.springframework.core.io.InputStreamResource;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RequestParam;

/**
 * Feign client for communication with the UserProfile DM service.
 */
public interface WorkspaceBsClient {


    /**
     * Creates a new empty document in the specified folder.
     *
     * @param folderId the ID of the folder in which to create the document
     * @param data     the DTO containing document details (e.g., name, description)
     * @return the DTO of the newly created document, including its generated ID and initial metadata
     * @throws feign.FeignException.FeignClientException.UnprocessableEntity if a document with the same name already exists
     * @throws feign.FeignException.FeignClientException.InternalServerError if an internal error occurs in the DM service
     */
    @PostMapping(path = "/workspace/folders/{folderId}/documents",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.CREATED)
    OutputCreatedDocumentDto createDocument(
            @PathVariable("folderId") @NotBlank String folderId,
            @RequestBody @Valid InputCreateDocumentDto data);

    /**
     * Replaces the content of an existing document, creating a new version.
     *
     * @param documentId the ID of the document to update
     * @param file       the new document file as InputStreamResource
     */
    @PutMapping(path = "/workspace/documents/{documentId}/content",
            consumes = MediaType.APPLICATION_OCTET_STREAM_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputUpdatedDocumentDto uploadNewContent(
            @PathVariable(name = "documentId") String documentId,
            @RequestParam(name = "fileName") String fileName,
            @RequestBody Resource file
    );


    @GetMapping(path = "/workspace/documents/{documentId}/publiclink", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    OutputDocumentPublicLinkDto getPublicLink(
            @PathVariable(name = "documentId") @NotBlank String documentId);


    /**
     * Retrieve the user's workspace
     */
    @GetMapping(path = "/workspace", produces = MediaType.APPLICATION_JSON_VALUE)
    WorkspaceDto getWorkspace();

    /**
     * Allow the folder creation in a user's workspace.
     *
     * @param folderId The ID of the parent folder.
     * @return a string containing the {folderId} of the newly created folder.
     */
    @PostMapping(
            path = "workspace/folders/{folderId}/folders",
            produces = MediaType.APPLICATION_JSON_VALUE,
            consumes = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.CREATED)
    String createFolderByFolderId(
            @PathVariable @NotBlank String folderId,
            @RequestBody InputFolderCreateDto inputFolderCreateDto);

    /**
     * Retrieves the folder in the parent folder id matching the provided name.
     *
     * @param folderId The ID of the parent folder.
     * @param name     The name of the folder to match.
     * @return A list of {@link FolderDto} representing the folders and sub-folders.
     */
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Folder found"),
            @ApiResponse(responseCode = "400", description = "Bad request"),
            @ApiResponse(responseCode = "404", description = "Folder not found")
    })
    @GetMapping(path = "workspace/folders/{folderId}/folder-by-name", produces = MediaType.APPLICATION_JSON_VALUE)
    FolderDto getFolderByParentFolderIdAndName(
            @PathVariable String folderId,
            @RequestParam String name
    );
}
