package com.finconsgroup.itserr.marketplace.userprofile.bs.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.enums.SortDirection;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2BusinessException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ExecutionException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.userprofile.bs.bean.UserProfileApplicationEvent;
import com.finconsgroup.itserr.marketplace.userprofile.bs.bean.UserProfileEndorsement;
import com.finconsgroup.itserr.marketplace.userprofile.bs.bean.UserProfileEndorsementApplicationEvent;
import com.finconsgroup.itserr.marketplace.userprofile.bs.bean.UserProfileInterests;
import com.finconsgroup.itserr.marketplace.userprofile.bs.bean.UserProfileInterestsApplicationEvent;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.SocialNetworkingD4SClient;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.UserProfileDmClient;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.WorkspaceBsClient;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputAddProjectToUserProfilesDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputAdminPatchUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputCreateDocumentDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputFindUserProfilesByInterestsDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputFindUserProfilesByPrincipalsDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputPatchUserProfileProjectDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputRemoveProjectFromUserProfilesDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.OutputAdminPatchUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.OutputCreatedDocumentDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.OutputDocumentPublicLinkDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.OutputUserProfileFolderDetails;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.OutputUsernameListD4SDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.FolderDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.InputCVDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.InputFolderCreateDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.InputPatchUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.InputUpdateUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.InputUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.OutputEndorsementAcknowledgementDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.OutputExpertiseDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.OutputPatchUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.OutputUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.OutputUserSettingDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.WorkspaceDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.exception.UserProfileAlreadyExistsException;
import com.finconsgroup.itserr.marketplace.userprofile.bs.exception.UserProfileNotFoundException;
import com.finconsgroup.itserr.marketplace.userprofile.bs.mapper.UserProfileSettingMapper;
import com.finconsgroup.itserr.marketplace.userprofile.bs.service.UserProfileService;
import com.finconsgroup.itserr.marketplace.userprofile.bs.util.Language;
import com.finconsgroup.itserr.marketplace.userprofile.bs.util.UniqueFilenameGenerator;
import feign.FeignException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.net.URI;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import static com.finconsgroup.itserr.marketplace.userprofile.bs.enums.MessagingEventType.CREATED;
import static com.finconsgroup.itserr.marketplace.userprofile.bs.enums.MessagingEventType.CV_UPDATED;
import static com.finconsgroup.itserr.marketplace.userprofile.bs.enums.MessagingEventType.ENDORSEMENT;
import static com.finconsgroup.itserr.marketplace.userprofile.bs.enums.MessagingEventType.IMAGE_UPDATED;
import static com.finconsgroup.itserr.marketplace.userprofile.bs.enums.MessagingEventType.INTERESTS;
import static com.finconsgroup.itserr.marketplace.userprofile.bs.enums.MessagingEventType.LANGUAGE_UPDATED;
import static com.finconsgroup.itserr.marketplace.userprofile.bs.enums.MessagingEventType.UPDATED;
import static com.finconsgroup.itserr.marketplace.userprofile.bs.util.Constants.HIDDEN_FOLDER_DESCRIPTION;
import static com.finconsgroup.itserr.marketplace.userprofile.bs.util.Constants.HIDDEN_FOLDER_HIDDEN_FLAG;
import static com.finconsgroup.itserr.marketplace.userprofile.bs.util.Constants.HIDDEN_FOLDER_NAME;
import static com.finconsgroup.itserr.marketplace.userprofile.bs.util.Constants.PERSONAL_CV;
import static com.finconsgroup.itserr.marketplace.userprofile.bs.util.Constants.PROFILE_IMAGE;

/**
 * Default implementation of {@link UserProfileService}
 * to perform operations related to UserProfile resources.
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultUserProfileService implements UserProfileService {

    public static final String D4S_RETRIEVE_USER_EXCEPTION = "Issue while retrieving users";
    private static final String D4S_RETRIEVE_USER_NOT_FOUND_MESSAGE = "HTTP 404";

    private final UserProfileDmClient userProfileDmClient;
    private final WorkspaceBsClient workspaceBsClient;
    private final SocialNetworkingD4SClient socialNetworkingD4SClient;
    private final UserProfileSettingMapper settingMapper;
    private final ApplicationEventPublisher applicationEventPublisher;

    @NonNull
    @Override
    public URI uploadImage(@NonNull MultipartFile image) {

        OutputUserProfileFolderDetails userProfileFolderDetails = profileFolderRetriever();
        String fileName = UniqueFilenameGenerator.generate(image.getOriginalFilename());

        InputCreateDocumentDto imageData = InputCreateDocumentDto.builder()
                .name(fileName)
                .description(PROFILE_IMAGE)
                .build();

        OutputCreatedDocumentDto documentDto =
                workspaceBsClient.createDocument(userProfileFolderDetails.getUserProfileFolderId().toString(), imageData);
        workspaceBsClient.uploadNewContent(documentDto.getId(), fileName, image.getResource());

        OutputDocumentPublicLinkDto outputDocumentPublicLinkDto = workspaceBsClient.getPublicLink(documentDto.getId());
        InputPatchUserProfileDto inputPatchUserProfileDto = InputPatchUserProfileDto.builder().imageUri(outputDocumentPublicLinkDto.getPublicLink()).build();

        OutputPatchUserProfileDto outputPatchUserProfileDto = userProfileDmClient.patchUserProfile(inputPatchUserProfileDto);
        applicationEventPublisher.publishEvent(new UserProfileApplicationEvent(userProfileDmClient.findById(), IMAGE_UPDATED));
        return outputPatchUserProfileDto.getImageUri();
    }


    @NonNull
    @Override
    public URI uploadCv(@NonNull MultipartFile cvFile) {
        OutputUserProfileFolderDetails userProfileFolderDetails = profileFolderRetriever();
        String fileName = UniqueFilenameGenerator.generate(cvFile.getOriginalFilename());

        InputCreateDocumentDto cvData = InputCreateDocumentDto.builder()
                .name(fileName)
                .description(PERSONAL_CV)
                .build();

        OutputCreatedDocumentDto documentDto =
                workspaceBsClient.createDocument(userProfileFolderDetails.getUserProfileFolderId().toString(), cvData);
        workspaceBsClient.uploadNewContent(documentDto.getId(), fileName, cvFile.getResource());

        OutputDocumentPublicLinkDto outputDocumentPublicLinkDto = workspaceBsClient.getPublicLink(documentDto.getId());
        InputPatchUserProfileDto inputPatchUserProfileDto = InputPatchUserProfileDto.builder()
                .inputCVDto(InputCVDto.builder()
                        .fileUrl(String.valueOf(outputDocumentPublicLinkDto.getPublicLink()))
                        .filename(cvData.getName())
                        .size(cvFile.getSize()).build()).build();

        OutputPatchUserProfileDto patchUserProfileDto = userProfileDmClient.patchUserProfile(inputPatchUserProfileDto);
        applicationEventPublisher.publishEvent(new UserProfileApplicationEvent(userProfileDmClient.findById(), CV_UPDATED));
        return URI.create(patchUserProfileDto.getOutputCVDto().getFileUrl());
    }


    private OutputUserProfileFolderDetails profileFolderRetriever() {
        return Optional.ofNullable(userProfileDmClient.getProfileFolder())
                .orElseThrow(() -> new WP2ResourceNotFoundException("Profile folder not exist"));
    }

    @NonNull
    @Override
    public OutputUserProfileDto create(@NonNull UUID userId, @NonNull InputUserProfileDto inputUserProfileDto) {
        try {
            final UUID folderId = getOrCreateHiddenFolder(userId);
            inputUserProfileDto.setUserProfileFolderId(folderId);
            OutputUserProfileDto outputUserProfileDto = userProfileDmClient.create(inputUserProfileDto);
            applicationEventPublisher.publishEvent(new UserProfileApplicationEvent(outputUserProfileDto, CREATED));

            UserProfileInterests userProfileInterests = UserProfileInterests
                    .builder()
                    .id(outputUserProfileDto.getId())
                    .name(outputUserProfileDto.getFirstName() + " " + outputUserProfileDto.getLastName())
                    .timestamp(outputUserProfileDto.getCreationTime().toInstant())
                    .build();

            List<String> expertises = Optional.ofNullable(outputUserProfileDto.getExpertises())
                    .orElse(List.of())
                    .stream()
                    .map(OutputExpertiseDto::getDisplayName)
                    .toList();

            publishInterestsEvent(userProfileInterests, expertises, Set.of(userId));

            return outputUserProfileDto;
        } catch (FeignException.Conflict e) {
            throw new UserProfileAlreadyExistsException(userId);
        } catch (Exception e) {
            throw new WP2ExecutionException(e);
        }
    }

    private UUID getOrCreateHiddenFolder(UUID userId) {
        WorkspaceDto workspace = getUserWorkspace();
        try {
            FolderDto folder = getUserProfileFolderId(workspace.getId());
            return UUID.fromString(folder.getId());
        } catch (FeignException.NotFound e) {
            String newFolderId = createUserProfileFolder(workspace.getId());
            log.info("Created hidden folder for user: {} with folder ID: {}", userId, newFolderId);
            return UUID.fromString(newFolderId);
        }
    }

    public @NonNull WorkspaceDto getUserWorkspace() {
        return workspaceBsClient.getWorkspace();
    }

    @Override
    public @NonNull OutputUserProfileDto getById(@NonNull UUID userId) {
        try {
            return userProfileDmClient.getById(userId);
        } catch (FeignException.NotFound e) {
            throw new UserProfileNotFoundException(userId);
        } catch (Exception e) {
            throw new WP2ExecutionException(e);
        }
    }

    @Override
    public @NonNull OutputUserProfileDto updateById(@NonNull UUID userId, @NonNull InputUpdateUserProfileDto inputUpdateUserProfileDto) {
        try {
            OutputUserProfileDto outputUserProfileDto = userProfileDmClient.updateById(inputUpdateUserProfileDto);
            applicationEventPublisher.publishEvent(new UserProfileApplicationEvent(outputUserProfileDto, UPDATED));
            return outputUserProfileDto;
        } catch (FeignException.NotFound e) {
            throw new UserProfileNotFoundException(userId);
        } catch (Exception e) {
            throw new WP2ExecutionException(e);
        }
    }

    @Override
    public @NonNull FolderDto getUserProfileFolderId(@NonNull String folderId) {
        return workspaceBsClient.getFolderByParentFolderIdAndName(folderId, HIDDEN_FOLDER_NAME);
    }

    @Override
    public @NonNull String createUserProfileFolder(@NonNull String parentFolderId) {
        return workspaceBsClient.createFolderByFolderId(parentFolderId, getHiddenFolderPayload());
    }

    @Override
    public @NonNull OutputUserSettingDto updateLanguage(String language) {
        language = Language.fromString(language).toString();
        InputPatchUserProfileDto inputPatchUserProfileDto = InputPatchUserProfileDto.builder().language(language).build();
        OutputPatchUserProfileDto patchUserProfileDto = userProfileDmClient.patchUserProfile(inputPatchUserProfileDto);
        applicationEventPublisher.publishEvent(new UserProfileApplicationEvent(userProfileDmClient.findById(), LANGUAGE_UPDATED));
        return settingMapper.patchedUserToSetting(patchUserProfileDto);

    }

    @NonNull
    @Override
    public List<OutputUserProfileDto> getUserProfilesByRole(String roleName) {
        OutputUsernameListD4SDto outputUsernameListD4SDto = null;
        try {
            outputUsernameListD4SDto = socialNetworkingD4SClient.getUsernamesByRole(roleName);
            if (outputUsernameListD4SDto == null || !outputUsernameListD4SDto.isSuccess()) {
                if (outputUsernameListD4SDto == null || StringUtils.isBlank(outputUsernameListD4SDto.getMessage())) {
                    throw new WP2BusinessException(D4S_RETRIEVE_USER_EXCEPTION);
                } else {
                    throw new WP2BusinessException(outputUsernameListD4SDto.getMessage());
                }
            }
        } catch (FeignException.InternalServerError e) {
            // in case of not found response, return an empty list
            // so that client can take appropriate action
            if (e.getMessage().contains(D4S_RETRIEVE_USER_NOT_FOUND_MESSAGE)) {
                log.error("Users not found for roleName - {}", roleName, e);
            } else {
                throw e;
            }
        }

        if (outputUsernameListD4SDto != null && outputUsernameListD4SDto.getResult() != null &&
                !outputUsernameListD4SDto.getResult().isEmpty()) {
            List<OutputUserProfileDto> outputUserProfileDtos = userProfileDmClient.findAllByPrincipals(
                    InputFindUserProfilesByPrincipalsDto.builder().principals(outputUsernameListD4SDto.getResult()).build(),
                    0,
                    outputUsernameListD4SDto.getResult().size(),
                    "id",
                    SortDirection.ASC
            ).getContent();

            Set<String> foundPrincipals = outputUserProfileDtos.stream().map(OutputUserProfileDto::getPreferredUsername)
                    .filter(StringUtils::isNotBlank).collect(Collectors.toSet());
            Set<String> missingPrincipals = new HashSet<>(outputUsernameListD4SDto.getResult());
            missingPrincipals.removeAll(foundPrincipals);
            if (!missingPrincipals.isEmpty()) {
                log.debug("UserProfiles not found following principals with role {} - {}", roleName, missingPrincipals);
            }
            return outputUserProfileDtos;
        }

        return List.of();
    }

    @Override
    @NonNull
    public void addProjectToUserProfiles(InputAddProjectToUserProfilesDto inputAddProjectToUserProfilesDto) {
        List<OutputUserProfileDto> outputUserProfileDtos = userProfileDmClient.addProjectToUserProfiles(inputAddProjectToUserProfilesDto);
        outputUserProfileDtos.forEach(outputUserProfileDto ->
                applicationEventPublisher.publishEvent(new UserProfileApplicationEvent(outputUserProfileDto, UPDATED))
        );
    }

    @Override
    @NonNull
    public void removeProjectFromUserProfiles(InputRemoveProjectFromUserProfilesDto inputRemoveProjectFromUserProfilesDto) {
        List<OutputUserProfileDto> outputUserProfileDtos = userProfileDmClient.removeProjectFromUserProfiles(inputRemoveProjectFromUserProfilesDto);
        outputUserProfileDtos.forEach(outputUserProfileDto ->
                applicationEventPublisher.publishEvent(new UserProfileApplicationEvent(outputUserProfileDto, UPDATED))
        );
    }

    @Override
    @NonNull
    public void patchUserProfileProject(InputPatchUserProfileProjectDto inputPatchUserProfileProjectDto) {
        List<OutputUserProfileDto> outputUserProfileDtos = userProfileDmClient.patchUserProfileProject(inputPatchUserProfileProjectDto);
        outputUserProfileDtos.forEach(outputUserProfileDto ->
                applicationEventPublisher.publishEvent(new UserProfileApplicationEvent(outputUserProfileDto, UPDATED))
        );
    }

    @Override
    @NonNull
    public OutputAdminPatchUserProfileDto patchUserProfile(UUID profileId, InputAdminPatchUserProfileDto inputAdminPatchUserProfileDto) {
        OutputAdminPatchUserProfileDto dto = userProfileDmClient.patchUserProfile(profileId, inputAdminPatchUserProfileDto);
        applicationEventPublisher.publishEvent(new UserProfileApplicationEvent(userProfileDmClient.getById(profileId), UPDATED));
        return dto;
    }

    @Override
    public void processUserProfileStatusChange(@NonNull UUID userId, @NonNull Boolean active) {
        if (active) {
            userProfileDmClient.activate();
        } else {
            userProfileDmClient.deactivate();
        }
        applicationEventPublisher.publishEvent(new UserProfileApplicationEvent(userProfileDmClient.getById(userId), UPDATED));
    }

    @Override
    public void publishAllProfilesMessages() {
        int pageNumber = 0;
        long fetchedCount = 0;
        long totalCount;
        while (true) {
            OutputPageDto<OutputUserProfileDto> userProfilesPage = userProfileDmClient.findAll(pageNumber, 50, "id", SortDirection.ASC);

            if (userProfilesPage.getContent() == null || userProfilesPage.getContent().isEmpty()) {
                break;
            }

            userProfilesPage.getContent().forEach(userProfileDto ->
                    applicationEventPublisher.publishEvent(new UserProfileApplicationEvent(userProfileDto, UPDATED))
            );

            totalCount = userProfilesPage.getPage().getTotalElements();
            fetchedCount += userProfilesPage.getContent().size();
            if (totalCount <= fetchedCount) {
                break;
            }

            pageNumber++;
        }
    }

    @Override
    public void publishInterestsEvent(@NonNull UserProfileInterests userProfileInterests,
                                      @Nullable List<String> stringsToCheck,
                                      @NonNull Set<UUID> userIdsToExclude) {

        if (stringsToCheck == null || stringsToCheck.isEmpty()) {
            return;
        }

        List<UUID> userProfileIdsWithInterests = userProfileDmClient.findMatchingInterests(
                InputFindUserProfilesByInterestsDto.builder()
                        .stringsToCheck(stringsToCheck)
                        .build()
        );

        // did not find any matching user profiles
        if (userProfileIdsWithInterests == null || userProfileIdsWithInterests.isEmpty()) {
            return;
        }

        userProfileInterests.setNotifyUserIds(new LinkedHashSet<>(userProfileIdsWithInterests));
        applicationEventPublisher.publishEvent(new UserProfileInterestsApplicationEvent(userProfileInterests, INTERESTS));
    }

    private InputFolderCreateDto getHiddenFolderPayload() {
        return InputFolderCreateDto.builder()
                .name(HIDDEN_FOLDER_NAME)
                .description(HIDDEN_FOLDER_DESCRIPTION)
                .hidden(HIDDEN_FOLDER_HIDDEN_FLAG)
                .build();
    }

    @Override
    public @NonNull Boolean updateHidePanel(Boolean hidePanel) {
        InputPatchUserProfileDto inputPatchUserProfileDto = InputPatchUserProfileDto.builder().hidePanel(hidePanel).build();
        OutputPatchUserProfileDto patchUserProfileDto = userProfileDmClient.patchUserProfile(inputPatchUserProfileDto);
        applicationEventPublisher.publishEvent(new UserProfileApplicationEvent(userProfileDmClient.findById(), UPDATED));
        return patchUserProfileDto.getHidePanel();
    }

    @Override
    @NonNull
    public OutputEndorsementAcknowledgementDto addEndorsement(UUID profileId, UUID expertiseId) {
        OutputEndorsementAcknowledgementDto dto = userProfileDmClient.addEndorsement(profileId, expertiseId);
        OutputUserProfileDto userProfileDto = userProfileDmClient.getById(profileId);

        applicationEventPublisher.publishEvent(new UserProfileApplicationEvent(userProfileDto, UPDATED));

        UserProfileEndorsement userProfileEndorsement = UserProfileEndorsement
                .builder()
                .id(dto.getId())
                .name(userProfileDto.getFirstName() + " " + userProfileDto.getLastName())
                .title(userProfileDto.getPreferredUsername())
                .notifyUserIds(Set.of(dto.getEndorsedId()))
                .expertiseId(dto.getExpertiseId())
                .expertiseDisplayName(dto.getExpertiseDisplayName())
                .timestamp(dto.getCreationTime())
                .build();

        applicationEventPublisher.publishEvent(new UserProfileEndorsementApplicationEvent(userProfileEndorsement, ENDORSEMENT));
        return dto;
    }

    @Override
    @NonNull
    public void removeEndorsement(UUID profileId, UUID expertiseId) {
        userProfileDmClient.removeEndorsement(profileId, expertiseId);
        applicationEventPublisher.publishEvent(new UserProfileApplicationEvent(userProfileDmClient.getById(profileId), UPDATED));
    }

}
