package com.finconsgroup.itserr.marketplace.userprofile.bs.util;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;

import lombok.NoArgsConstructor;
import org.apache.commons.io.FilenameUtils;

/**
 * Utility class for generating unique filenames by appending a timestamp.
 */
@NoArgsConstructor
public final class UniqueFilenameGenerator {

    private static final DateTimeFormatter TIMESTAMP_FORMATTER =
            DateTimeFormatter.ofPattern("yyyyMMddHHmmss");

    /**
     * Generates a unique filename by appending a timestamp to the base name of the original filename.
     * The format is: baseName_yyyyMMddHHmmss[.extension]
     *
     * @param originalFilename the original filename (e.g., "avatar.png")
     * @return the generated unique filename (e.g., "avatar_20250527153045.png")
     * @throws IllegalArgumentException if originalFilename is null or empty
     */
    public static String generate(String originalFilename) {
        if (originalFilename == null || originalFilename.trim().isEmpty()) {
            throw new IllegalArgumentException("Original filename must not be null or empty");
        }

        String baseName = FilenameUtils.getBaseName(originalFilename);
        String extension = FilenameUtils.getExtension(originalFilename);
        String timestamp = LocalDateTime.now().format(TIMESTAMP_FORMATTER);

        if (extension.isEmpty()) {
            return String.format("%s_%s", baseName, timestamp);
        }
        return String.format("%s_%s.%s", baseName, timestamp, extension);
    }
}
