/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.util.EnumSet;
import org.apache.hadoop.fs.CreateFlag;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileContext;
import org.apache.hadoop.fs.FileContextTestHelper;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.Options;
import org.apache.hadoop.fs.ParentNotDirectoryException;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.permission.FsPermission;
import org.junit.After;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Before;
import org.junit.Test;

public abstract class FileContextSymlinkBaseTest {
    static final long seed = 3735928559L;
    static final int blockSize = 8192;
    static final int fileSize = 16384;
    protected static FileContext fc;

    protected abstract String getScheme();

    protected abstract String testBaseDir1() throws IOException;

    protected abstract String testBaseDir2() throws IOException;

    protected abstract URI testURI();

    protected IOException unwrapException(IOException e) {
        return e;
    }

    protected static void createAndWriteFile(FileContext fc, Path p) throws IOException {
        FileContextTestHelper.createFile(fc, p, 2, new Options.CreateOpts[]{Options.CreateOpts.createParent(), Options.CreateOpts.repFac((short)1), Options.CreateOpts.blockSize((long)8192L)});
    }

    protected static void createAndWriteFile(Path p) throws IOException {
        FileContextSymlinkBaseTest.createAndWriteFile(fc, p);
    }

    protected static void readFile(Path p) throws IOException {
        FileContextTestHelper.readFile(fc, p, 16384);
    }

    protected static void readFile(FileContext fc, Path p) throws IOException {
        FileContextTestHelper.readFile(fc, p, 16384);
    }

    protected static void appendToFile(Path p) throws IOException {
        FileContextTestHelper.appendToFile(fc, p, 2, new Options.CreateOpts[]{Options.CreateOpts.blockSize((long)8192L)});
    }

    @Before
    public void setUp() throws Exception {
        fc.mkdir(new Path(this.testBaseDir1()), FileContext.DEFAULT_PERM, true);
        fc.mkdir(new Path(this.testBaseDir2()), FileContext.DEFAULT_PERM, true);
    }

    @After
    public void tearDown() throws Exception {
        fc.delete(new Path(this.testBaseDir1()), true);
        fc.delete(new Path(this.testBaseDir2()), true);
    }

    @Test
    public void testStatRoot() throws IOException {
        Assert.assertFalse((boolean)fc.getFileLinkStatus(new Path("/")).isSymlink());
    }

    @Test
    public void testSetWDNotResolvesLinks() throws IOException {
        Path dir = new Path(this.testBaseDir1());
        Path linkToDir = new Path(this.testBaseDir1() + "/link");
        fc.createSymlink(dir, linkToDir, false);
        fc.setWorkingDirectory(linkToDir);
        Assert.assertEquals((Object)linkToDir.getName(), (Object)fc.getWorkingDirectory().getName());
    }

    @Test
    public void testCreateDanglingLink() throws IOException {
        Path file = new Path("/noSuchFile");
        Path link = new Path(this.testBaseDir1() + "/link");
        fc.createSymlink(file, link, false);
        try {
            fc.getFileStatus(link);
            Assert.fail((String)"Got file status of non-existant file");
        }
        catch (FileNotFoundException fileNotFoundException) {
            // empty catch block
        }
        fc.delete(link, false);
    }

    @Test
    public void testCreateLinkToNullEmpty() throws IOException {
        Path link = new Path(this.testBaseDir1() + "/link");
        try {
            fc.createSymlink(null, link, false);
            Assert.fail((String)"Can't create symlink to null");
        }
        catch (NullPointerException e) {
            // empty catch block
        }
        try {
            fc.createSymlink(new Path(""), link, false);
            Assert.fail((String)"Can't create symlink to empty string");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    @Test
    public void testCreateLinkCanCreateParent() throws IOException {
        Path file = new Path(this.testBaseDir1() + "/file");
        Path link = new Path(this.testBaseDir2() + "/linkToFile");
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        fc.delete(new Path(this.testBaseDir2()), true);
        try {
            fc.createSymlink(file, link, false);
            Assert.fail((String)"Created link without first creating parent dir");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        Assert.assertFalse((boolean)FileContextTestHelper.exists(fc, new Path(this.testBaseDir2())));
        fc.createSymlink(file, link, true);
        FileContextSymlinkBaseTest.readFile(link);
    }

    @Test
    public void testMkdirExistingLink() throws IOException {
        Path dir = new Path(this.testBaseDir1() + "/link");
        fc.createSymlink(new Path("/doesNotExist"), dir, false);
        try {
            fc.mkdir(dir, FileContext.DEFAULT_PERM, false);
            Assert.fail((String)"Created a dir where a symlink exists");
        }
        catch (FileAlreadyExistsException e) {
        }
        catch (IOException e) {
            Assert.assertEquals((Object)"file", (Object)this.getScheme());
        }
    }

    @Test
    public void testCreateFileViaDanglingLinkParent() throws IOException {
        Path dir = new Path(this.testBaseDir1() + "/dangling");
        Path file = new Path(this.testBaseDir1() + "/dangling/file");
        fc.createSymlink(new Path("/doesNotExist"), dir, false);
        try {
            FSDataOutputStream out = fc.create(file, EnumSet.of(CreateFlag.CREATE), new Options.CreateOpts[]{Options.CreateOpts.repFac((short)1), Options.CreateOpts.blockSize((long)8192L)});
            out.close();
            Assert.fail((String)"Created a link with dangling link parent");
        }
        catch (FileNotFoundException e) {
            // empty catch block
        }
    }

    @Test
    public void testDeleteLink() throws IOException {
        Path file = new Path(this.testBaseDir1() + "/file");
        Path link = new Path(this.testBaseDir1() + "/linkToFile");
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        fc.createSymlink(file, link, false);
        FileContextSymlinkBaseTest.readFile(link);
        fc.delete(link, false);
        try {
            FileContextSymlinkBaseTest.readFile(link);
            Assert.fail((String)"Symlink should have been deleted");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        fc.createSymlink(file, link, false);
    }

    @Test
    public void testOpenResolvesLinks() throws IOException {
        Path file = new Path(this.testBaseDir1() + "/noSuchFile");
        Path link = new Path(this.testBaseDir1() + "/link");
        fc.createSymlink(file, link, false);
        try {
            fc.open(link);
            Assert.fail((String)"link target does not exist");
        }
        catch (FileNotFoundException fileNotFoundException) {
            // empty catch block
        }
        fc.delete(link, false);
    }

    @Test
    public void testStatLinkToFile() throws IOException {
        Path file = new Path(this.testBaseDir1() + "/file");
        Path linkToFile = new Path(this.testBaseDir1() + "/linkToFile");
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        fc.createSymlink(file, linkToFile, false);
        Assert.assertFalse((boolean)fc.getFileLinkStatus(linkToFile).isDirectory());
        Assert.assertTrue((boolean)FileContextTestHelper.isSymlink(fc, linkToFile));
        Assert.assertTrue((boolean)FileContextTestHelper.isFile(fc, linkToFile));
        Assert.assertFalse((boolean)FileContextTestHelper.isDir(fc, linkToFile));
        Assert.assertEquals((Object)file.toUri().getPath(), (Object)fc.getLinkTarget(linkToFile).toString());
        if (!"file".equals(this.getScheme())) {
            Assert.assertEquals((Object)fc.getFileStatus(file), (Object)fc.getFileStatus(linkToFile));
            Assert.assertEquals((Object)fc.makeQualified(file), (Object)fc.getFileStatus(linkToFile).getPath());
            Assert.assertEquals((Object)fc.makeQualified(linkToFile), (Object)fc.getFileLinkStatus(linkToFile).getPath());
        }
    }

    @Test
    public void testStatRelLinkToFile() throws IOException {
        Assume.assumeTrue((!"file".equals(this.getScheme()) ? 1 : 0) != 0);
        Path baseDir = new Path(this.testBaseDir1());
        Path file = new Path(this.testBaseDir1(), "file");
        Path linkToFile = new Path(this.testBaseDir1(), "linkToFile");
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        fc.createSymlink(new Path("file"), linkToFile, false);
        Assert.assertEquals((Object)fc.getFileStatus(file), (Object)fc.getFileStatus(linkToFile));
        Assert.assertEquals((Object)fc.makeQualified(file), (Object)fc.getFileStatus(linkToFile).getPath());
        Assert.assertEquals((Object)fc.makeQualified(linkToFile), (Object)fc.getFileLinkStatus(linkToFile).getPath());
    }

    @Test
    public void testStatLinkToDir() throws IOException {
        Path dir = new Path(this.testBaseDir1());
        Path linkToDir = new Path(this.testBaseDir1() + "/linkToDir");
        fc.createSymlink(dir, linkToDir, false);
        Assert.assertFalse((boolean)fc.getFileStatus(linkToDir).isSymlink());
        Assert.assertTrue((boolean)FileContextTestHelper.isDir(fc, linkToDir));
        Assert.assertFalse((boolean)fc.getFileLinkStatus(linkToDir).isDirectory());
        Assert.assertTrue((boolean)fc.getFileLinkStatus(linkToDir).isSymlink());
        Assert.assertFalse((boolean)FileContextTestHelper.isFile(fc, linkToDir));
        Assert.assertTrue((boolean)FileContextTestHelper.isDir(fc, linkToDir));
        Assert.assertEquals((Object)dir.toUri().getPath(), (Object)fc.getLinkTarget(linkToDir).toString());
    }

    @Test
    public void testStatDanglingLink() throws IOException {
        Path file = new Path("/noSuchFile");
        Path link = new Path(this.testBaseDir1() + "/link");
        fc.createSymlink(file, link, false);
        Assert.assertFalse((boolean)fc.getFileLinkStatus(link).isDirectory());
        Assert.assertTrue((boolean)fc.getFileLinkStatus(link).isSymlink());
    }

    @Test
    public void testStatNonExistantFiles() throws IOException {
        Path fileAbs = new Path("/doesNotExist");
        try {
            fc.getFileLinkStatus(fileAbs);
            Assert.fail((String)"Got FileStatus for non-existant file");
        }
        catch (FileNotFoundException f) {
            // empty catch block
        }
        try {
            fc.getLinkTarget(fileAbs);
            Assert.fail((String)"Got link target for non-existant file");
        }
        catch (FileNotFoundException fileNotFoundException) {
            // empty catch block
        }
    }

    @Test
    public void testStatNonLinks() throws IOException {
        Path dir = new Path(this.testBaseDir1());
        Path file = new Path(this.testBaseDir1() + "/file");
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        try {
            fc.getLinkTarget(dir);
            Assert.fail((String)"Lstat'd a non-symlink");
        }
        catch (IOException e) {
            // empty catch block
        }
        try {
            fc.getLinkTarget(file);
            Assert.fail((String)"Lstat'd a non-symlink");
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    @Test
    public void testRecursiveLinks() throws IOException {
        Path link1 = new Path(this.testBaseDir1() + "/link1");
        Path link2 = new Path(this.testBaseDir1() + "/link2");
        fc.createSymlink(link1, link2, false);
        fc.createSymlink(link2, link1, false);
        try {
            FileContextSymlinkBaseTest.readFile(link1);
            Assert.fail((String)"Read recursive link");
        }
        catch (FileNotFoundException f) {
        }
        catch (IOException x) {
            Assert.assertEquals((Object)("Possible cyclic loop while following symbolic link " + link1.toString()), (Object)x.getMessage());
        }
    }

    private void checkLink(Path linkAbs, Path expectedTarget, Path targetQual) throws IOException {
        Path dir = new Path(this.testBaseDir1());
        Assert.assertTrue((boolean)FileContextTestHelper.isFile(fc, linkAbs));
        Assert.assertFalse((boolean)FileContextTestHelper.isDir(fc, linkAbs));
        Assert.assertFalse((boolean)fc.getFileStatus(linkAbs).isSymlink());
        Assert.assertFalse((boolean)fc.getFileStatus(linkAbs).isDirectory());
        Assert.assertEquals((long)16384L, (long)fc.getFileStatus(linkAbs).getLen());
        Assert.assertTrue((boolean)FileContextTestHelper.isSymlink(fc, linkAbs));
        Assert.assertFalse((boolean)fc.getFileLinkStatus(linkAbs).isDirectory());
        Assert.assertEquals((Object)targetQual.toString(), (Object)fc.getFileLinkStatus(linkAbs).getSymlink().toString());
        Assert.assertEquals((Object)targetQual, (Object)fc.getFileLinkStatus(linkAbs).getSymlink());
        if (!"file".equals(this.getScheme())) {
            FileContext localFc = FileContext.getLocalFSFileContext();
            Path linkQual = new Path(this.testURI().toString(), linkAbs);
            Assert.assertEquals((Object)targetQual, (Object)localFc.getFileLinkStatus(linkQual).getSymlink());
        }
        Assert.assertEquals((Object)expectedTarget, (Object)fc.getLinkTarget(linkAbs));
        fc.setWorkingDirectory(dir);
        FileContextSymlinkBaseTest.readFile(new Path("linkToFile"));
        FileContextSymlinkBaseTest.readFile(linkAbs);
        FileContextSymlinkBaseTest.readFile(new Path(this.testURI().toString(), linkAbs));
        boolean failureExpected = !"file".equals(this.getScheme());
        try {
            FileContextSymlinkBaseTest.readFile(new Path(this.getScheme() + "://" + this.testBaseDir1() + "/linkToFile"));
            Assert.assertFalse((boolean)failureExpected);
        }
        catch (Exception e) {
            Assert.assertTrue((boolean)failureExpected);
        }
        if (!"file".equals(this.getScheme())) {
            FileContext localFc = FileContext.getLocalFSFileContext();
            FileContextSymlinkBaseTest.readFile(localFc, new Path(this.testURI().toString(), linkAbs));
        }
    }

    @Test
    public void testCreateLinkUsingRelPaths() throws IOException {
        Path fileAbs = new Path(this.testBaseDir1(), "file");
        Path linkAbs = new Path(this.testBaseDir1(), "linkToFile");
        Path schemeAuth = new Path(this.testURI().toString());
        Path fileQual = new Path(schemeAuth, this.testBaseDir1() + "/file");
        FileContextSymlinkBaseTest.createAndWriteFile(fileAbs);
        fc.setWorkingDirectory(new Path(this.testBaseDir1()));
        fc.createSymlink(new Path("file"), new Path("linkToFile"), false);
        this.checkLink(linkAbs, new Path("file"), fileQual);
        Path dir1 = new Path(this.testBaseDir1());
        Path dir2 = new Path(this.testBaseDir2());
        Path linkViaDir2 = new Path(this.testBaseDir2(), "linkToFile");
        Path fileViaDir2 = new Path(schemeAuth, this.testBaseDir2() + "/file");
        fc.rename(dir1, dir2, new Options.Rename[]{Options.Rename.OVERWRITE});
        Assert.assertEquals((Object)fileViaDir2, (Object)fc.getFileLinkStatus(linkViaDir2).getSymlink());
        FileContextSymlinkBaseTest.readFile(linkViaDir2);
    }

    @Test
    public void testCreateLinkUsingAbsPaths() throws IOException {
        Path fileAbs = new Path(this.testBaseDir1() + "/file");
        Path linkAbs = new Path(this.testBaseDir1() + "/linkToFile");
        Path schemeAuth = new Path(this.testURI().toString());
        Path fileQual = new Path(schemeAuth, this.testBaseDir1() + "/file");
        FileContextSymlinkBaseTest.createAndWriteFile(fileAbs);
        fc.createSymlink(fileAbs, linkAbs, false);
        this.checkLink(linkAbs, fileAbs, fileQual);
        Path dir1 = new Path(this.testBaseDir1());
        Path dir2 = new Path(this.testBaseDir2());
        Path linkViaDir2 = new Path(this.testBaseDir2(), "linkToFile");
        fc.rename(dir1, dir2, new Options.Rename[]{Options.Rename.OVERWRITE});
        Assert.assertEquals((Object)fileQual, (Object)fc.getFileLinkStatus(linkViaDir2).getSymlink());
        try {
            FileContextSymlinkBaseTest.readFile(linkViaDir2);
            Assert.fail((String)"The target should not exist");
        }
        catch (FileNotFoundException x) {
            // empty catch block
        }
    }

    @Test
    public void testCreateLinkUsingFullyQualPaths() throws IOException {
        Path fileAbs = new Path(this.testBaseDir1(), "file");
        Path linkAbs = new Path(this.testBaseDir1(), "linkToFile");
        Path fileQual = new Path(this.testURI().toString(), fileAbs);
        Path linkQual = new Path(this.testURI().toString(), linkAbs);
        FileContextSymlinkBaseTest.createAndWriteFile(fileAbs);
        fc.createSymlink(fileQual, linkQual, false);
        this.checkLink(linkAbs, "file".equals(this.getScheme()) ? fileAbs : fileQual, fileQual);
        Path dir1 = new Path(this.testBaseDir1());
        Path dir2 = new Path(this.testBaseDir2());
        Path linkViaDir2 = new Path(this.testBaseDir2(), "linkToFile");
        fc.rename(dir1, dir2, new Options.Rename[]{Options.Rename.OVERWRITE});
        Assert.assertEquals((Object)fileQual, (Object)fc.getFileLinkStatus(linkViaDir2).getSymlink());
        try {
            FileContextSymlinkBaseTest.readFile(linkViaDir2);
            Assert.fail((String)"The target should not exist");
        }
        catch (FileNotFoundException x) {
            // empty catch block
        }
    }

    @Test
    public void testCreateLinkUsingPartQualPath1() throws IOException {
        Assume.assumeTrue((!"file".equals(this.getScheme()) ? 1 : 0) != 0);
        Path schemeAuth = new Path(this.testURI().toString());
        Path fileWoHost = new Path(this.getScheme() + "://" + this.testBaseDir1() + "/file");
        Path link = new Path(this.testBaseDir1() + "/linkToFile");
        Path linkQual = new Path(schemeAuth, this.testBaseDir1() + "/linkToFile");
        FileContext localFc = FileContext.getLocalFSFileContext();
        fc.createSymlink(fileWoHost, link, false);
        Assert.assertEquals((Object)fileWoHost, (Object)fc.getLinkTarget(linkQual));
        Assert.assertEquals((Object)fileWoHost.toString(), (Object)fc.getFileLinkStatus(link).getSymlink().toString());
        Assert.assertEquals((Object)fileWoHost.toString(), (Object)fc.getFileLinkStatus(linkQual).getSymlink().toString());
        Assert.assertEquals((Object)fileWoHost.toString(), (Object)localFc.getFileLinkStatus(linkQual).getSymlink().toString());
        try {
            FileContextSymlinkBaseTest.readFile(link);
            Assert.fail((String)"DFS requires URIs with schemes have an authority");
        }
        catch (RuntimeException e) {
            // empty catch block
        }
    }

    @Test
    public void testCreateLinkUsingPartQualPath2() throws IOException {
        Path link = new Path(this.testBaseDir1(), "linkToFile");
        Path fileWoScheme = new Path("//" + this.testURI().getAuthority() + this.testBaseDir1() + "/file");
        if ("file".equals(this.getScheme())) {
            return;
        }
        fc.createSymlink(fileWoScheme, link, false);
        Assert.assertEquals((Object)fileWoScheme, (Object)fc.getLinkTarget(link));
        Assert.assertEquals((Object)fileWoScheme.toString(), (Object)fc.getFileLinkStatus(link).getSymlink().toString());
        try {
            FileContextSymlinkBaseTest.readFile(link);
            Assert.fail((String)"Accessed a file with w/o scheme");
        }
        catch (IOException e) {
            Assert.assertEquals((Object)"No AbstractFileSystem for scheme: null", (Object)e.getMessage());
        }
    }

    @Test
    public void testLinkStatusAndTargetWithNonLink() throws IOException {
        Path schemeAuth = new Path(this.testURI().toString());
        Path dir = new Path(this.testBaseDir1());
        Path dirQual = new Path(schemeAuth, dir.toString());
        Path file = new Path(this.testBaseDir1(), "file");
        Path fileQual = new Path(schemeAuth, file.toString());
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        Assert.assertEquals((Object)fc.getFileStatus(file), (Object)fc.getFileLinkStatus(file));
        Assert.assertEquals((Object)fc.getFileStatus(dir), (Object)fc.getFileLinkStatus(dir));
        try {
            fc.getLinkTarget(file);
            Assert.fail((String)"Get link target on non-link should throw an IOException");
        }
        catch (IOException x) {
            Assert.assertEquals((Object)("Path " + fileQual + " is not a symbolic link"), (Object)x.getMessage());
        }
        try {
            fc.getLinkTarget(dir);
            Assert.fail((String)"Get link target on non-link should throw an IOException");
        }
        catch (IOException x) {
            Assert.assertEquals((Object)("Path " + dirQual + " is not a symbolic link"), (Object)x.getMessage());
        }
    }

    @Test
    public void testCreateLinkToDirectory() throws IOException {
        Path dir1 = new Path(this.testBaseDir1());
        Path file = new Path(this.testBaseDir1(), "file");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        fc.createSymlink(dir1, linkToDir, false);
        Assert.assertFalse((boolean)FileContextTestHelper.isFile(fc, linkToDir));
        Assert.assertTrue((boolean)FileContextTestHelper.isDir(fc, linkToDir));
        Assert.assertTrue((boolean)fc.getFileStatus(linkToDir).isDirectory());
        Assert.assertTrue((boolean)fc.getFileLinkStatus(linkToDir).isSymlink());
    }

    @Test
    public void testCreateFileViaSymlink() throws IOException {
        Path dir = new Path(this.testBaseDir1());
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        Path fileViaLink = new Path(linkToDir, "file");
        fc.createSymlink(dir, linkToDir, false);
        FileContextSymlinkBaseTest.createAndWriteFile(fileViaLink);
        Assert.assertTrue((boolean)FileContextTestHelper.isFile(fc, fileViaLink));
        Assert.assertFalse((boolean)FileContextTestHelper.isDir(fc, fileViaLink));
        Assert.assertFalse((boolean)fc.getFileLinkStatus(fileViaLink).isSymlink());
        Assert.assertFalse((boolean)fc.getFileStatus(fileViaLink).isDirectory());
        FileContextSymlinkBaseTest.readFile(fileViaLink);
        fc.delete(fileViaLink, true);
        Assert.assertFalse((boolean)FileContextTestHelper.exists(fc, fileViaLink));
    }

    @Test
    public void testCreateDirViaSymlink() throws IOException {
        Path dir1 = new Path(this.testBaseDir1());
        Path subDir = new Path(this.testBaseDir1(), "subDir");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        Path subDirViaLink = new Path(linkToDir, "subDir");
        fc.createSymlink(dir1, linkToDir, false);
        fc.mkdir(subDirViaLink, FileContext.DEFAULT_PERM, true);
        Assert.assertTrue((boolean)FileContextTestHelper.isDir(fc, subDirViaLink));
        fc.delete(subDirViaLink, false);
        Assert.assertFalse((boolean)FileContextTestHelper.exists(fc, subDirViaLink));
        Assert.assertFalse((boolean)FileContextTestHelper.exists(fc, subDir));
    }

    @Test
    public void testCreateLinkViaLink() throws IOException {
        Path dir1 = new Path(this.testBaseDir1());
        Path file = new Path(this.testBaseDir1(), "file");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        Path fileViaLink = new Path(linkToDir, "file");
        Path linkToFile = new Path(linkToDir, "linkToFile");
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        fc.createSymlink(dir1, linkToDir, false);
        fc.createSymlink(fileViaLink, linkToFile, false);
        Assert.assertTrue((boolean)FileContextTestHelper.isFile(fc, linkToFile));
        Assert.assertTrue((boolean)fc.getFileLinkStatus(linkToFile).isSymlink());
        FileContextSymlinkBaseTest.readFile(linkToFile);
        Assert.assertEquals((long)16384L, (long)fc.getFileStatus(linkToFile).getLen());
        Assert.assertEquals((Object)fileViaLink, (Object)fc.getLinkTarget(linkToFile));
    }

    @Test
    public void testListStatusUsingLink() throws IOException {
        Path file = new Path(this.testBaseDir1(), "file");
        Path link = new Path(this.testBaseDir1(), "link");
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        fc.createSymlink(new Path(this.testBaseDir1()), link, false);
        FileStatus[] stats = fc.util().listStatus(link);
        Assert.assertTrue((stats.length == 2 || stats.length == 3 ? 1 : 0) != 0);
        RemoteIterator statsItor = fc.listStatus(link);
        int dirLen = 0;
        while (statsItor.hasNext()) {
            statsItor.next();
            ++dirLen;
        }
        Assert.assertTrue((dirLen == 2 || dirLen == 3 ? 1 : 0) != 0);
    }

    @Test
    public void testCreateLinkTwice() throws IOException {
        Path file = new Path(this.testBaseDir1(), "file");
        Path link = new Path(this.testBaseDir1(), "linkToFile");
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        fc.createSymlink(file, link, false);
        try {
            fc.createSymlink(file, link, false);
            Assert.fail((String)"link already exists");
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    @Test
    public void testCreateLinkToLink() throws IOException {
        Path dir1 = new Path(this.testBaseDir1());
        Path file = new Path(this.testBaseDir1(), "file");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        Path linkToLink = new Path(this.testBaseDir2(), "linkToLink");
        Path fileViaLink = new Path(this.testBaseDir2(), "linkToLink/file");
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        fc.createSymlink(dir1, linkToDir, false);
        fc.createSymlink(linkToDir, linkToLink, false);
        Assert.assertTrue((boolean)FileContextTestHelper.isFile(fc, fileViaLink));
        Assert.assertFalse((boolean)FileContextTestHelper.isDir(fc, fileViaLink));
        Assert.assertFalse((boolean)fc.getFileLinkStatus(fileViaLink).isSymlink());
        Assert.assertFalse((boolean)fc.getFileStatus(fileViaLink).isDirectory());
        FileContextSymlinkBaseTest.readFile(fileViaLink);
    }

    @Test
    public void testCreateFileDirExistingLink() throws IOException {
        Path file = new Path(this.testBaseDir1(), "file");
        Path link = new Path(this.testBaseDir1(), "linkToFile");
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        fc.createSymlink(file, link, false);
        try {
            FileContextSymlinkBaseTest.createAndWriteFile(link);
            Assert.fail((String)"link already exists");
        }
        catch (IOException x) {
            // empty catch block
        }
        try {
            fc.mkdir(link, FsPermission.getDefault(), false);
            Assert.fail((String)"link already exists");
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    @Test
    public void testUseLinkAferDeleteLink() throws IOException {
        Path file = new Path(this.testBaseDir1(), "file");
        Path link = new Path(this.testBaseDir1(), "linkToFile");
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        fc.createSymlink(file, link, false);
        fc.delete(link, false);
        try {
            FileContextSymlinkBaseTest.readFile(link);
            Assert.fail((String)"link was deleted");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        FileContextSymlinkBaseTest.readFile(file);
        fc.createSymlink(file, link, false);
        FileContextSymlinkBaseTest.readFile(link);
    }

    @Test
    public void testCreateLinkToDot() throws IOException {
        Path dir = new Path(this.testBaseDir1());
        Path file = new Path(this.testBaseDir1(), "file");
        Path link = new Path(this.testBaseDir1(), "linkToDot");
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        fc.setWorkingDirectory(dir);
        try {
            fc.createSymlink(new Path("."), link, false);
            Assert.fail((String)"Created symlink to dot");
        }
        catch (IOException x) {
            // empty catch block
        }
    }

    @Test
    public void testCreateLinkToDotDot() throws IOException {
        Path file = new Path(this.testBaseDir1(), "test/file");
        Path dotDot = new Path(this.testBaseDir1(), "test/..");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        Path fileViaLink = new Path(linkToDir, "test/file");
        Assert.assertEquals((Object)this.testBaseDir1(), (Object)dotDot.toString());
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        fc.createSymlink(dotDot, linkToDir, false);
        FileContextSymlinkBaseTest.readFile(fileViaLink);
        Assert.assertEquals((long)16384L, (long)fc.getFileStatus(fileViaLink).getLen());
    }

    @Test
    public void testCreateLinkToDotDotPrefix() throws IOException {
        Path file = new Path(this.testBaseDir1(), "file");
        Path dir = new Path(this.testBaseDir1(), "test");
        Path link = new Path(this.testBaseDir1(), "test/link");
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        fc.mkdir(dir, FsPermission.getDefault(), false);
        fc.setWorkingDirectory(dir);
        fc.createSymlink(new Path("../file"), link, false);
        FileContextSymlinkBaseTest.readFile(link);
        Assert.assertEquals((Object)new Path("../file"), (Object)fc.getLinkTarget(link));
    }

    @Test
    public void testRenameFileViaSymlink() throws IOException {
        Path dir = new Path(this.testBaseDir1());
        Path file = new Path(this.testBaseDir1(), "file");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        Path fileViaLink = new Path(linkToDir, "file");
        Path fileNewViaLink = new Path(linkToDir, "fileNew");
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        fc.createSymlink(dir, linkToDir, false);
        fc.rename(fileViaLink, fileNewViaLink, new Options.Rename[0]);
        Assert.assertFalse((boolean)FileContextTestHelper.exists(fc, fileViaLink));
        Assert.assertFalse((boolean)FileContextTestHelper.exists(fc, file));
        Assert.assertTrue((boolean)FileContextTestHelper.exists(fc, fileNewViaLink));
    }

    @Test
    public void testRenameFileToDestViaSymlink() throws IOException {
        Path dir = new Path(this.testBaseDir1());
        Path file = new Path(this.testBaseDir1(), "file");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        Path subDir = new Path(linkToDir, "subDir");
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        fc.createSymlink(dir, linkToDir, false);
        fc.mkdir(subDir, FileContext.DEFAULT_PERM, false);
        try {
            fc.rename(file, subDir, new Options.Rename[0]);
            Assert.fail((String)"Renamed file to a directory");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof IOException));
        }
        Assert.assertTrue((boolean)FileContextTestHelper.exists(fc, file));
    }

    @Test
    public void testRenameDirViaSymlink() throws IOException {
        Path baseDir = new Path(this.testBaseDir1());
        Path dir = new Path(baseDir, "dir");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        Path dirViaLink = new Path(linkToDir, "dir");
        Path dirNewViaLink = new Path(linkToDir, "dirNew");
        fc.mkdir(dir, FileContext.DEFAULT_PERM, false);
        fc.createSymlink(baseDir, linkToDir, false);
        Assert.assertTrue((boolean)FileContextTestHelper.exists(fc, dirViaLink));
        fc.rename(dirViaLink, dirNewViaLink, new Options.Rename[0]);
        Assert.assertFalse((boolean)FileContextTestHelper.exists(fc, dirViaLink));
        Assert.assertFalse((boolean)FileContextTestHelper.exists(fc, dir));
        Assert.assertTrue((boolean)FileContextTestHelper.exists(fc, dirNewViaLink));
    }

    @Test
    public void testRenameSymlinkViaSymlink() throws IOException {
        Path baseDir = new Path(this.testBaseDir1());
        Path file = new Path(this.testBaseDir1(), "file");
        Path link = new Path(this.testBaseDir1(), "link");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        Path linkViaLink = new Path(linkToDir, "link");
        Path linkNewViaLink = new Path(linkToDir, "linkNew");
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        fc.createSymlink(file, link, false);
        fc.createSymlink(baseDir, linkToDir, false);
        fc.rename(linkViaLink, linkNewViaLink, new Options.Rename[0]);
        Assert.assertFalse((boolean)FileContextTestHelper.exists(fc, linkViaLink));
        Assert.assertTrue((boolean)FileContextTestHelper.exists(fc, file));
        Assert.assertTrue((boolean)fc.getFileLinkStatus(linkNewViaLink).isSymlink());
        FileContextSymlinkBaseTest.readFile(linkNewViaLink);
    }

    @Test
    public void testRenameDirToSymlinkToDir() throws IOException {
        Path dir1 = new Path(this.testBaseDir1());
        Path subDir = new Path(this.testBaseDir2(), "subDir");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        fc.mkdir(subDir, FileContext.DEFAULT_PERM, false);
        fc.createSymlink(subDir, linkToDir, false);
        try {
            fc.rename(dir1, linkToDir, new Options.Rename[]{Options.Rename.OVERWRITE});
            Assert.fail((String)"Renamed directory to a symlink");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof IOException));
        }
        Assert.assertTrue((boolean)FileContextTestHelper.exists(fc, dir1));
        Assert.assertTrue((boolean)FileContextTestHelper.exists(fc, linkToDir));
    }

    @Test
    public void testRenameDirToSymlinkToFile() throws IOException {
        Path dir1 = new Path(this.testBaseDir1());
        Path file = new Path(this.testBaseDir2(), "file");
        Path linkToFile = new Path(this.testBaseDir2(), "linkToFile");
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        fc.createSymlink(file, linkToFile, false);
        try {
            fc.rename(dir1, linkToFile, new Options.Rename[]{Options.Rename.OVERWRITE});
            Assert.fail((String)"Renamed directory to a symlink");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof IOException));
        }
        Assert.assertTrue((boolean)FileContextTestHelper.exists(fc, dir1));
        Assert.assertTrue((boolean)FileContextTestHelper.exists(fc, linkToFile));
    }

    @Test
    public void testRenameDirToDanglingSymlink() throws IOException {
        Path dir = new Path(this.testBaseDir1());
        Path link = new Path(this.testBaseDir2(), "linkToFile");
        fc.createSymlink(new Path("/doesNotExist"), link, false);
        try {
            fc.rename(dir, link, new Options.Rename[]{Options.Rename.OVERWRITE});
            Assert.fail((String)"Renamed directory to a symlink");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof IOException));
        }
        Assert.assertTrue((boolean)FileContextTestHelper.exists(fc, dir));
        Assert.assertTrue((fc.getFileLinkStatus(link) != null ? 1 : 0) != 0);
    }

    @Test
    public void testRenameFileToSymlinkToDir() throws IOException {
        Path file = new Path(this.testBaseDir1(), "file");
        Path subDir = new Path(this.testBaseDir1(), "subDir");
        Path link = new Path(this.testBaseDir1(), "link");
        fc.mkdir(subDir, FileContext.DEFAULT_PERM, false);
        fc.createSymlink(subDir, link, false);
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        try {
            fc.rename(file, link, new Options.Rename[0]);
            Assert.fail((String)"Renamed file to symlink w/o overwrite");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof FileAlreadyExistsException));
        }
        fc.rename(file, link, new Options.Rename[]{Options.Rename.OVERWRITE});
        Assert.assertFalse((boolean)FileContextTestHelper.exists(fc, file));
        Assert.assertTrue((boolean)FileContextTestHelper.exists(fc, link));
        Assert.assertTrue((boolean)FileContextTestHelper.isFile(fc, link));
        Assert.assertFalse((boolean)fc.getFileLinkStatus(link).isSymlink());
    }

    @Test
    public void testRenameFileToSymlinkToFile() throws IOException {
        Path file1 = new Path(this.testBaseDir1(), "file1");
        Path file2 = new Path(this.testBaseDir1(), "file2");
        Path link = new Path(this.testBaseDir1(), "linkToFile");
        FileContextSymlinkBaseTest.createAndWriteFile(file1);
        FileContextSymlinkBaseTest.createAndWriteFile(file2);
        fc.createSymlink(file2, link, false);
        try {
            fc.rename(file1, link, new Options.Rename[0]);
            Assert.fail((String)"Renamed file to symlink w/o overwrite");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof FileAlreadyExistsException));
        }
        fc.rename(file1, link, new Options.Rename[]{Options.Rename.OVERWRITE});
        Assert.assertFalse((boolean)FileContextTestHelper.exists(fc, file1));
        Assert.assertTrue((boolean)FileContextTestHelper.exists(fc, link));
        Assert.assertTrue((boolean)FileContextTestHelper.isFile(fc, link));
        Assert.assertFalse((boolean)fc.getFileLinkStatus(link).isSymlink());
    }

    @Test
    public void testRenameFileToDanglingSymlink() throws IOException {
        if ("file".equals(this.getScheme())) {
            return;
        }
        Path file1 = new Path(this.testBaseDir1(), "file1");
        Path link = new Path(this.testBaseDir1(), "linkToFile");
        FileContextSymlinkBaseTest.createAndWriteFile(file1);
        fc.createSymlink(new Path("/doesNotExist"), link, false);
        try {
            fc.rename(file1, link, new Options.Rename[0]);
        }
        catch (IOException iOException) {
            // empty catch block
        }
        fc.rename(file1, link, new Options.Rename[]{Options.Rename.OVERWRITE});
        Assert.assertFalse((boolean)FileContextTestHelper.exists(fc, file1));
        Assert.assertTrue((boolean)FileContextTestHelper.exists(fc, link));
        Assert.assertTrue((boolean)FileContextTestHelper.isFile(fc, link));
        Assert.assertFalse((boolean)fc.getFileLinkStatus(link).isSymlink());
    }

    @Test
    public void testRenameSymlinkNonExistantDest() throws IOException {
        Path file = new Path(this.testBaseDir1(), "file");
        Path link1 = new Path(this.testBaseDir1(), "linkToFile1");
        Path link2 = new Path(this.testBaseDir1(), "linkToFile2");
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        fc.createSymlink(file, link1, false);
        fc.rename(link1, link2, new Options.Rename[0]);
        Assert.assertTrue((boolean)fc.getFileLinkStatus(link2).isSymlink());
        FileContextSymlinkBaseTest.readFile(link2);
        FileContextSymlinkBaseTest.readFile(file);
        Assert.assertFalse((boolean)FileContextTestHelper.exists(fc, link1));
    }

    @Test
    public void testRenameSymlinkToExistingFile() throws IOException {
        Path file1 = new Path(this.testBaseDir1(), "file");
        Path file2 = new Path(this.testBaseDir1(), "someFile");
        Path link = new Path(this.testBaseDir1(), "linkToFile");
        FileContextSymlinkBaseTest.createAndWriteFile(file1);
        FileContextSymlinkBaseTest.createAndWriteFile(file2);
        fc.createSymlink(file2, link, false);
        try {
            fc.rename(link, file1, new Options.Rename[0]);
            Assert.fail((String)"Renamed w/o passing overwrite");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof FileAlreadyExistsException));
        }
        fc.rename(link, file1, new Options.Rename[]{Options.Rename.OVERWRITE});
        Assert.assertFalse((boolean)FileContextTestHelper.exists(fc, link));
        Assert.assertTrue((boolean)fc.getFileLinkStatus(file1).isSymlink());
        Assert.assertEquals((Object)file2, (Object)fc.getLinkTarget(file1));
    }

    @Test
    public void testRenameSymlinkToExistingDir() throws IOException {
        Path dir1 = new Path(this.testBaseDir1());
        Path dir2 = new Path(this.testBaseDir2());
        Path subDir = new Path(this.testBaseDir2(), "subDir");
        Path link = new Path(this.testBaseDir1(), "linkToDir");
        fc.createSymlink(dir1, link, false);
        try {
            fc.rename(link, dir2, new Options.Rename[0]);
            Assert.fail((String)"Renamed link to a directory");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof IOException));
        }
        try {
            fc.rename(link, dir2, new Options.Rename[]{Options.Rename.OVERWRITE});
            Assert.fail((String)"Renamed link to a directory");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof IOException));
        }
        fc.mkdir(subDir, FsPermission.getDefault(), false);
        try {
            fc.rename(link, dir2, new Options.Rename[]{Options.Rename.OVERWRITE});
            Assert.fail((String)"Renamed link to a directory");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof IOException));
        }
    }

    @Test
    public void testRenameSymlinkToItself() throws IOException {
        Path link = new Path(this.testBaseDir1(), "linkToFile1");
        fc.createSymlink(new Path("/doestNotExist"), link, false);
        try {
            fc.rename(link, link, new Options.Rename[0]);
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof FileAlreadyExistsException));
        }
        try {
            fc.rename(link, link, new Options.Rename[]{Options.Rename.OVERWRITE});
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof FileAlreadyExistsException));
        }
    }

    @Test
    public void testRenameSymlink() throws IOException {
        Path file = new Path(this.testBaseDir1(), "file");
        Path link1 = new Path(this.testBaseDir1(), "linkToFile1");
        Path link2 = new Path(this.testBaseDir1(), "linkToFile2");
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        fc.createSymlink(file, link1, false);
        fc.rename(link1, link2, new Options.Rename[0]);
        Assert.assertTrue((boolean)fc.getFileLinkStatus(link2).isSymlink());
        Assert.assertFalse((boolean)fc.getFileStatus(link2).isDirectory());
        FileContextSymlinkBaseTest.readFile(link2);
        FileContextSymlinkBaseTest.readFile(file);
        try {
            FileContextSymlinkBaseTest.createAndWriteFile(link2);
            Assert.fail((String)"link was not renamed");
        }
        catch (IOException x) {
            // empty catch block
        }
    }

    @Test
    public void testRenameSymlinkToFileItLinksTo() throws IOException {
        if ("file".equals(this.getScheme())) {
            return;
        }
        Path file = new Path(this.testBaseDir1(), "file");
        Path link = new Path(this.testBaseDir1(), "linkToFile");
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        fc.createSymlink(file, link, false);
        try {
            fc.rename(link, file, new Options.Rename[0]);
            Assert.fail((String)"Renamed symlink to its target");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof FileAlreadyExistsException));
        }
        Assert.assertTrue((boolean)FileContextTestHelper.isFile(fc, file));
        Assert.assertTrue((boolean)FileContextTestHelper.exists(fc, link));
        Assert.assertTrue((boolean)FileContextTestHelper.isSymlink(fc, link));
        Assert.assertEquals((Object)file, (Object)fc.getLinkTarget(link));
        try {
            fc.rename(link, file, new Options.Rename[]{Options.Rename.OVERWRITE});
            Assert.fail((String)"Renamed symlink to its target");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof FileAlreadyExistsException));
        }
        Assert.assertTrue((boolean)FileContextTestHelper.isFile(fc, file));
        Assert.assertTrue((boolean)FileContextTestHelper.exists(fc, link));
        Assert.assertTrue((boolean)FileContextTestHelper.isSymlink(fc, link));
        Assert.assertEquals((Object)file, (Object)fc.getLinkTarget(link));
    }

    @Test
    public void testRenameSymlinkToDirItLinksTo() throws IOException {
        if ("file".equals(this.getScheme())) {
            return;
        }
        Path dir = new Path(this.testBaseDir1(), "dir");
        Path link = new Path(this.testBaseDir1(), "linkToDir");
        fc.mkdir(dir, FileContext.DEFAULT_PERM, false);
        fc.createSymlink(dir, link, false);
        try {
            fc.rename(link, dir, new Options.Rename[0]);
            Assert.fail((String)"Renamed symlink to its target");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof FileAlreadyExistsException));
        }
        Assert.assertTrue((boolean)FileContextTestHelper.isDir(fc, dir));
        Assert.assertTrue((boolean)FileContextTestHelper.exists(fc, link));
        Assert.assertTrue((boolean)FileContextTestHelper.isSymlink(fc, link));
        Assert.assertEquals((Object)dir, (Object)fc.getLinkTarget(link));
        try {
            fc.rename(link, dir, new Options.Rename[]{Options.Rename.OVERWRITE});
            Assert.fail((String)"Renamed symlink to its target");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof FileAlreadyExistsException));
        }
        Assert.assertTrue((boolean)FileContextTestHelper.isDir(fc, dir));
        Assert.assertTrue((boolean)FileContextTestHelper.exists(fc, link));
        Assert.assertTrue((boolean)FileContextTestHelper.isSymlink(fc, link));
        Assert.assertEquals((Object)dir, (Object)fc.getLinkTarget(link));
    }

    @Test
    public void testRenameLinkTarget() throws IOException {
        Path file = new Path(this.testBaseDir1(), "file");
        Path fileNew = new Path(this.testBaseDir1(), "fileNew");
        Path link = new Path(this.testBaseDir1(), "linkToFile");
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        fc.createSymlink(file, link, false);
        fc.rename(file, fileNew, new Options.Rename[]{Options.Rename.OVERWRITE});
        try {
            FileContextSymlinkBaseTest.readFile(link);
            Assert.fail((String)"Link should be dangling");
        }
        catch (IOException x) {
            // empty catch block
        }
        fc.rename(fileNew, file, new Options.Rename[]{Options.Rename.OVERWRITE});
        FileContextSymlinkBaseTest.readFile(link);
    }

    @Test
    public void testRenameFileWithDestParentSymlink() throws IOException {
        Path link = new Path(this.testBaseDir1(), "link");
        Path file1 = new Path(this.testBaseDir1(), "file1");
        Path file2 = new Path(this.testBaseDir1(), "file2");
        Path file3 = new Path(link, "file3");
        Path dir2 = new Path(this.testBaseDir2());
        fc.createSymlink(dir2, link, false);
        FileContextSymlinkBaseTest.createAndWriteFile(file1);
        fc.rename(file1, file3, new Options.Rename[0]);
        Assert.assertFalse((boolean)FileContextTestHelper.exists(fc, file1));
        Assert.assertTrue((boolean)FileContextTestHelper.exists(fc, file3));
        fc.rename(file3, file1, new Options.Rename[0]);
        fc.delete(link, false);
        fc.createSymlink(file2, link, false);
        try {
            fc.rename(file1, file3, new Options.Rename[0]);
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof FileNotFoundException));
        }
        FileContextSymlinkBaseTest.createAndWriteFile(file2);
        try {
            fc.rename(file1, file3, new Options.Rename[0]);
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)(this.unwrapException(e) instanceof ParentNotDirectoryException));
        }
    }

    @Test
    public void testAccessFileViaInterSymlinkAbsTarget() throws IOException {
        Path baseDir = new Path(this.testBaseDir1());
        Path file = new Path(this.testBaseDir1(), "file");
        Path fileNew = new Path(baseDir, "fileNew");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        Path fileViaLink = new Path(linkToDir, "file");
        Path fileNewViaLink = new Path(linkToDir, "fileNew");
        fc.createSymlink(baseDir, linkToDir, false);
        FileContextSymlinkBaseTest.createAndWriteFile(fileViaLink);
        Assert.assertTrue((boolean)FileContextTestHelper.exists(fc, fileViaLink));
        Assert.assertTrue((boolean)FileContextTestHelper.isFile(fc, fileViaLink));
        Assert.assertFalse((boolean)FileContextTestHelper.isDir(fc, fileViaLink));
        Assert.assertFalse((boolean)fc.getFileLinkStatus(fileViaLink).isSymlink());
        Assert.assertFalse((boolean)FileContextTestHelper.isDir(fc, fileViaLink));
        Assert.assertEquals((Object)fc.getFileStatus(file), (Object)fc.getFileLinkStatus(file));
        Assert.assertEquals((Object)fc.getFileStatus(fileViaLink), (Object)fc.getFileLinkStatus(fileViaLink));
        FileContextSymlinkBaseTest.readFile(fileViaLink);
        FileContextSymlinkBaseTest.appendToFile(fileViaLink);
        fc.rename(fileViaLink, fileNewViaLink, new Options.Rename[0]);
        Assert.assertFalse((boolean)FileContextTestHelper.exists(fc, fileViaLink));
        Assert.assertTrue((boolean)FileContextTestHelper.exists(fc, fileNewViaLink));
        FileContextSymlinkBaseTest.readFile(fileNewViaLink);
        Assert.assertEquals((long)fc.getFileBlockLocations(fileNew, 0L, 1L).length, (long)fc.getFileBlockLocations(fileNewViaLink, 0L, 1L).length);
        Assert.assertEquals((Object)fc.getFileChecksum(fileNew), (Object)fc.getFileChecksum(fileNewViaLink));
        fc.delete(fileNewViaLink, true);
        Assert.assertFalse((boolean)FileContextTestHelper.exists(fc, fileNewViaLink));
    }

    @Test
    public void testAccessFileViaInterSymlinkQualTarget() throws IOException {
        Path baseDir = new Path(this.testBaseDir1());
        Path file = new Path(this.testBaseDir1(), "file");
        Path fileNew = new Path(baseDir, "fileNew");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        Path fileViaLink = new Path(linkToDir, "file");
        Path fileNewViaLink = new Path(linkToDir, "fileNew");
        fc.createSymlink(fc.makeQualified(baseDir), linkToDir, false);
        FileContextSymlinkBaseTest.createAndWriteFile(fileViaLink);
        Assert.assertEquals((Object)fc.getFileStatus(file), (Object)fc.getFileLinkStatus(file));
        Assert.assertEquals((Object)fc.getFileStatus(fileViaLink), (Object)fc.getFileLinkStatus(fileViaLink));
        FileContextSymlinkBaseTest.readFile(fileViaLink);
    }

    @Test
    public void testAccessFileViaInterSymlinkRelTarget() throws IOException {
        Assume.assumeTrue((!"file".equals(this.getScheme()) ? 1 : 0) != 0);
        Path baseDir = new Path(this.testBaseDir1());
        Path dir = new Path(this.testBaseDir1(), "dir");
        Path file = new Path(dir, "file");
        Path linkToDir = new Path(this.testBaseDir1(), "linkToDir");
        Path fileViaLink = new Path(linkToDir, "file");
        fc.mkdir(dir, FileContext.DEFAULT_PERM, false);
        fc.createSymlink(new Path("dir"), linkToDir, false);
        FileContextSymlinkBaseTest.createAndWriteFile(fileViaLink);
        Assert.assertEquals((Object)fc.makeQualified(file), (Object)fc.getFileStatus(file).getPath());
        Assert.assertEquals((Object)fc.getFileStatus(file), (Object)fc.getFileLinkStatus(file));
        Assert.assertEquals((Object)fc.getFileStatus(fileViaLink), (Object)fc.getFileLinkStatus(fileViaLink));
        Assert.assertEquals((Object)fc.getFileStatus(fileViaLink), (Object)fc.getFileLinkStatus(file));
    }

    @Test
    public void testAccessDirViaSymlink() throws IOException {
        Path baseDir = new Path(this.testBaseDir1());
        Path dir = new Path(this.testBaseDir1(), "dir");
        Path linkToDir = new Path(this.testBaseDir2(), "linkToDir");
        Path dirViaLink = new Path(linkToDir, "dir");
        fc.createSymlink(baseDir, linkToDir, false);
        fc.mkdir(dirViaLink, FileContext.DEFAULT_PERM, true);
        Assert.assertTrue((boolean)fc.getFileStatus(dirViaLink).isDirectory());
        FileStatus[] stats = fc.util().listStatus(dirViaLink);
        Assert.assertEquals((long)0L, (long)stats.length);
        RemoteIterator statsItor = fc.listStatus(dirViaLink);
        Assert.assertFalse((boolean)statsItor.hasNext());
        fc.delete(dirViaLink, false);
        Assert.assertFalse((boolean)FileContextTestHelper.exists(fc, dirViaLink));
        Assert.assertFalse((boolean)FileContextTestHelper.exists(fc, dir));
    }

    @Test
    public void testSetTimes() throws IOException {
        Path file = new Path(this.testBaseDir1(), "file");
        Path link = new Path(this.testBaseDir1(), "linkToFile");
        FileContextSymlinkBaseTest.createAndWriteFile(file);
        fc.createSymlink(file, link, false);
        long at = fc.getFileLinkStatus(link).getAccessTime();
        fc.setTimes(link, 2L, 3L);
        if (!"file".equals(this.getScheme())) {
            Assert.assertEquals((long)at, (long)fc.getFileLinkStatus(link).getAccessTime());
            Assert.assertEquals((long)3L, (long)fc.getFileStatus(file).getAccessTime());
            Assert.assertEquals((long)2L, (long)fc.getFileStatus(file).getModificationTime());
        }
    }
}

